/*
  Macro           HiliteAll
  Author          Carlo.Hogeveen@xs4all.nl
  Compatibility   Windows TSE v4       upwards
                  Linux   TSE v4.41.35 upwards
  Version         1.1.2   23 Sep 2024


  FUNCTIONALITY

  - When a Find or Replace command is started with its own key and hilites its
    match, then all other matches currently on the screen are hilited too.
  - It does so as a side effect to existing Find and Replace keys without
    interfering with their functionality.
  - It can hilite once, or keep hiliting the last search while you continue
    editing.

  Strong points
  - It also hilites during a Replace's (Yes/No/...) prompts.
  - You can configure separate hiliting colours for the extra matches.
  - It hilites a Replace with the Replace search options.

  Weak point
  - It only hilites for commands started with a configured key.


  INSTALLATION

  Copy this macro file to TSE's "mac" folder, and compile it,
  for instance by opening the file in TSE and using the Macro Compile menu.

  Execute the macro to configure it:
  - Mandatory, otherwise nothing happens:
    - You must configure for which keys extra Find/Replace hiliting should
      occur.
      Users of TSE Pro versions AFTER v4.4 get a quick configuration-start menu
      option to add auto-configurable Find and Replace keys first.
      This option has a limited capability, so you should check the result
      and maybe add some keys yourself too.
      When doing so, look for example in TSE's Search menu and/or .ui file
      to see what some of your existing Find and Replace keys are.
      You can configure as many or as few keys as you like.
      You might have to restart TSE for newly configured keys to take effect.
      The macro disables itself when no Find and Replace keys are configured.
  - Optional:
    - Configure hilite duration: "Once" or "Perpetual".
      With "Perpetual" you can go on editing a file and keep seeing results
      hilited based on the last search, which is really cool.
    - If "Perpetual", then configure which key(s) should stop the hiliting.
      Stopping hiliting is implemented as a side effect too; here it does not
      matter whether the Stop key(s) already exist or not.
      When you configure "Perpetual" without a Stop key, then you get a
      proposal to auto-install the single <Alt> key as a Stop key.
    - You can configure a separate colour for the extra hiliting matches.
      Advice: Do so, it separates the extra matches from the search's own
      match, and it makes stopping hiliting work better, and it looks cool.



  KNOWN IMPERFECTIONS

  Limited unhiliting during a Replace

    During a Replace you might unhilite the extra hiliting by using one of the
    "Stop keys" you configured, but only within these limitations:
    - Unhiliting only works if your Replace hiliting colour differs from TSE's
      Hilite, Block and CursorInBlock colours.
    - Unhiliting is done back to the default text colour.
      This means that syntax hiliting is not immediately restored.

  Replace-hiliting does not work for long file names

    This macro problem is based on a TSE problem.

    The only way that the macro can recognize, that it is in the per-occurrence
    question stage of a Replace action, is if there is a "   Replace (Yes/No/"
    string with the colour of a message in the status line.

    However, a Replace action displays the current file name BEFORE that string.

    So for a very long file name the question "    Replace (Yes/No/" disappears
    past the end of the status line, and the macro does no extra hiliting.

    Note that in such a case a user does not get to see the question either.

  Ignored search options:

    a n v +   These rightly ignored search options do not apply when hiliting
              all matches that are visible in the current editing window.

    <number>  Implementing this would take an insane amount of time.

    l         Extra hiliting does not happen for the text already hilited by
              the Find or Replace command. Therefore block marked text is not
              hilited either, because TSE's CUAmark macro uses block marking
              for hiliting. If block marked text is not hilited, then it makes
              some sense to ignore the l(ocal) search option altogether. Hmm.

  No hiliting for commands with a multiple-key combination

    For instance in the WordStar user interface ws.ui,
    the Find() is attached to the <ctrl q><f> key combination.

    This macro cannot hilite Finds in that case.

  Hiliting line numbers and window borders

    If TSE's configuration option to show line numbers is on, and a text is
    scrolled to the right so that a hilited word disappears beneath a line
    number, then the line number is hilited.

    Likewise, when a to be hilited string would occur to the left or right of
    the visible part of the text, the window borders are hilited. This one is
    a bit worse, because a hilited window border persists longer.

    I would like to solve these two minor issues in some future release.



  WISH LIST
  - Do not hilite line numbers and window borders.
  - Hiliting for Compressed View.
  - Hiliting for Incremental Search.
  - Sjoerd 28 Apr 2019:
    "My preferred configuration after a Replace: highlighting with different
     color of the processed text, no matter if the text has changed or not."



  DESIGN CHOICES

  Except for the above exceptions I chose to hilite as closely as possibly
  to the actually given command (a find or replace) and its search parameters.

  Alas, from the macro language it is not possible to automatically determine
  which of the two kinds of commands it was. Not reliably.

  Therefore a macro can not automatically determine whether to use the
  last find-search-options or the last replace-search-options.

  My chosen solution is to let the user explicitly assign a command-type
  to the keys they use to initiate a find-command or a replace-command.

  The downside is that the extra hiliting only occurs and changes after
  commands initiated by such a command-type specified key.

  The upside is more accurate search results that are more correctly hilited.

  A side benefit is, that you can configure which search commands should do the
  extra hiliting all and which ones should not.

  If you are in any way dissatisfied with my choices, then you might also want
  to try someone else's completely different approach at http://bit.ly/2XFRicw .



  HISTORY

  0.1 - 23 Apr 2019
  - Initial beta release.

  0.2 - 24 Apr 2019
  - You now get an error message if TSE's tse.ini configuration file is not
    writable. Thanks to Stein Oiestad for the report.
  - Added configurable Find and Replace hiliting colours.
  - Version was not displayed in the configuration menu.

  0.3 - 25 Apr 2019
  - Small bugs fixed, tiny improvements made.
  - This macro was not backwards compatible down to TSE v4.0, as it claimed,
    because it uses the TSE v4.4 StartPgm() command. This command is now
    emulated for older TSE versions down to TSE v4.0. Problem solved.

  0.4 - 25 Apr 2019
  - Improved adherence to the specific search-command, -string and -options.
  - The "stop" keys now immidiately unhilite the existing extra hiliting.

  0.4.1 - 25 Apr 2019
  - Fixed an erroneous helpline for "Duration" in the configuration menu.
    Thanks to Stein Oiestad for the report.

  0.4.2 - 26 Apr 2019
  - Only improved the documentation a bit.

  0.5 - 27 Apr 2019
  - Hilite All during Replace too.

  0.6 - 27 Apr 2019
  - Implemented the "c", "^" and "$" search options too.
  - Documented not hiliting regular expressions that also have an empty result.
  - Made the macro more future-proof by implementing QueryEditState()'s new
    built-in constant names, as well as backwards compatible with TSE v4.0.

  0.7 - 28 Apr 2019
  - Reformatted the configuration menu to clarify its functionality.
  - Faster hiliting for the per-match Replace questions.
  - Made the configuration request to enter a new key more conspicuous.
  - More logical colour names in the colour picker.
    I used to follow the Semware documentation, but because "intense" is a
    background colour property, I changed colour names like this:
    "intense bright red on yellow" to "bright red on intense yellow".

  0.8 - 29 Apr 2019
  - Simplified configuration by automatically updating the Macro AutoLoad List
    based on whether any keys are configured.
  - Made the screen less crowded when configuring Find, Replace and Stop keys
    by temporarily removing the main configuration menu.
  - Made configurable what to do after a Replace is finished:
      "Keep hiliting with the Replace parameters" (the new default),
      "Back to hiliting with the Find parameters", or
      "Stop hiliting" (what the previous version did).
  - Changed the default hiliting duration from "Once" to "Perpetual".

  0.9 - 30 Apr 2019
  - Now showing configured keys in the main menu too.
  - Tiny bugs fixed.

  0.9.1 - 2 May 2019
  - Replace used Find's configured colour instead of its own.

  0.9.2 - 3 May 2019
  - Fixed incorrect hiliting for hard expanded tabs
    both before and inside a search result.

  0.10 - 4 May 2019
  - Added limited capability to auto-configure keys (TSE Pro 4.4 upwards).

  1.0 - 11 May 2019
  - Release.
  - Just documentation tweaks.

  1.0.1 - 15 May 2019
  - Fixed what is mainly a documentation error:
    Auto-configurable-keys is not possible from TSE Pro v4.4 upwards,
    but only for TSE versions after TSE Pro v4.4 onwards.

  1.0.2 - 16 Oct 2019
  - Tiny maintenance update:
    Replaced the compare_versions() procedure with its v2.0 version, which
    recognizes more version formats.

  1.0.3 - 4 Apr 2020
  - Made the macro compile in Linux, and when run made it give a user friendly
    error message that the macro does not work in Linux. (The underlying problem
    is that the macro language's _idle_ hook is not implemented in Linux.)

  1.1   - 1 May 2020
  - Added Linux TSE Beta v4.41.35 compatibility, made possible by Semware
    fixing the _IDLE_ hook and a GetStrAttrXY() bug in Linux TSE.
  - Made the "Add auto-configurable Find and Replace keys" configuration option
    backwards compatible down to TSE Pro v4.0.

  1.1.1   17 Sep 2022
    Fixed incompatibility with TSE's '-i' command line option
    and the TSELOADDIR environment variable.

  1.1.2   23 Sep 2024
    Give a proper warning if writing to file tseload.dat fails.

*/





// Compatibility restrictions and mitigations

/*
  When compiled with a TSE version below TSE 4.0 the compiler reports
  a syntax error and hilights the first applicable line below.
*/

#ifdef LINUX
  #define WIN32 FALSE
  string SLASH [1] = '/'
#else
  string SLASH [1] = '\'
#endif

#ifdef WIN32
#else
   16-bit versions of TSE are not supported. You need at least TSE 4.0.
#endif

#ifdef EDITOR_VERSION
#else
   Editor Version is older than TSE 3.0. You need at least TSE 4.0.
#endif

#if EDITOR_VERSION < 4000h
   Editor Version is older than TSE 4.0. You need at least TSE 4.0.
#endif



#if EDITOR_VERSION > 4400h
  #define STATE_TWOKEY            _STATE_TWOKEY_
  #define STATE_WARN              _STATE_WARN_
  #define STATE_EDITOR_PAUSED     _STATE_EDITOR_PAUSED_
  #define STATE_POPWINDOW         _STATE_POPWINDOW_
  #define STATE_PROCESS_IN_WINDOW _STATE_PROCESS_IN_WINDOW_
  #define STATE_MENU              _STATE_MENU_
  #define STATE_PROMPTED          _STATE_PROMPTED_
#else
  #define STATE_TWOKEY            0x0200
  #define STATE_WARN              0x0400
  #define STATE_EDITOR_PAUSED     0x0800
  #define STATE_POPWINDOW         0x1000
  #define STATE_PROCESS_IN_WINDOW 0x2000
  #define STATE_MENU              0x4000
  #define STATE_PROMPTED          0x3000
#endif



#if EDITOR_VERSION < 4400h
  /*
    isAutoLoaded() 1.0

    This procedure implements TSE 4.4's isAutoLoaded() function for older TSE
    versions. This procedure differs in that here no parameter is allowed,
    so it can only examine the currently running macro's autoload status.
  */
  integer isAutoLoaded_id                    = 0
  string  isAutoLoaded_file_name_chrset [44] = "- !#$%&'()+,.0-9;=@A-Z[]^_`a-z{}~\d127-\d255"
  integer proc isAutoLoaded()
    string  autoload_name [255] = ''
    string  old_wordset    [32] = Set(WordSet, ChrSet(isAutoLoaded_file_name_chrset))
    integer org_id              = GetBufferId()
    integer result              = FALSE
    if isAutoLoaded_id
      GotoBufferId(isAutoLoaded_id)
    else
      autoload_name = SplitPath(CurrMacroFilename(), _NAME_) + ':isAutoLoaded'
      isAutoLoaded_id   = GetBufferId(autoload_name)
      if isAutoLoaded_id
        GotoBufferId(isAutoLoaded_id)
      else
        isAutoLoaded_id = CreateTempBuffer()
        ChangeCurrFilename(autoload_name, _DONT_PROMPT_|_DONT_EXPAND_|_OVERWRITE_)
      endif
    endif
    LoadBuffer(LoadDir() + 'tseload.dat')
    result = lFind(SplitPath(CurrMacroFilename(), _NAME_), "giw")
    Set(WordSet, old_wordset)
    GotoBufferId(org_id)
    return(result)
  end isAutoLoaded
#endif



#if EDITOR_VERSION < 4400h
  /*
    StartPgm()  1.0

    Below the line is StartPgm's TSE v4.4 documentation.

    When you make your macro backwards compatible downto TSE v4.0,
    then all four parameters become mandatory for all calls to StartPgm().

    If you do not need the 2nd, 3rd and 4th parameter, then you must use
    their default values: '', '' and _DEFAULT_.

    Tip:
      A practical use for StartPgm is that it lets you "start" a data file too,
      *if* Windows knows what program to run for the file's file type.

      For instance, you can "start" a URL, and StartPgm will start your default
      web browser for that URL.

    ---------------------------------------------------------------------------

    StartPgm

    Runs a program using the Windows ShellExecute function.

    Syntax:     INTEGER StartPgm(STRING pgm_name [, STRING args
                        [, STRING start_dir [, INTEGER startup_flags]]])

                - pgm_name is the name of the program to run.

                - args are optional command line arguments that should be passed
                  to pgm_name.

                - start_dir is an optional starting directory.

                - startup_flags are optional flags that control how pgm_name is
                  started.  Values can be: _START_HIDDEN_, _START_MINIMIZED_,
                  _START_MAXIMIZED_.

    Returns:    Non-zero if successful; zero (FALSE) on failure.

    Notes:      This function is the preferred way to run Win32 GUI programs
                from the editor.

    Examples:

                //Cause the editor to run g32.exe, editing the file "some.file"
                StartPgm("g32.exe", "some.file")

    See Also:   lDos(), Dos(), Shell()
  */
  #define SW_HIDE             0
  #define SW_SHOWNORMAL       1
  #define SW_NORMAL           1
  #define SW_SHOWMINIMIZED    2
  #define SW_SHOWMAXIMIZED    3
  #define SW_MAXIMIZE         3
  #define SW_SHOWNOACTIVATE   4
  #define SW_SHOW             5
  #define SW_MINIMIZE         6
  #define SW_SHOWMINNOACTIVE  7
  #define SW_SHOWNA           8
  #define SW_RESTORE          9
  #define SW_SHOWDEFAULT      10
  #define SW_MAX              10

  dll "<shell32.dll>"
    integer proc ShellExecute(
      integer h,          // handle to parent window
      string op:cstrval,  // specifies operation to perform
      string file:cstrval,// filename string
      string parm:cstrval,// specifies executable-file parameters
      string dir:cstrval, // specifies default directory
      integer show)       // whether file is shown when opened
      :"ShellExecuteA"
  end

  integer proc StartPgm(string  pgm_name,
                        string  args,
                        string  start_dir,
                        integer startup_flags)
    integer result              = FALSE
    integer return_code         = 0
    integer shell_startup_flags = 0
    case startup_flags
      when _DEFAULT_
        shell_startup_flags = SW_SHOWNORMAL
      when _START_HIDDEN_
        shell_startup_flags = SW_HIDE
      when _START_MAXIMIZED_
        shell_startup_flags = SW_SHOWMAXIMIZED
      when _START_MINIMIZED_
        shell_startup_flags = SW_SHOWMINIMIZED
      otherwise
        shell_startup_flags = SW_SHOWNORMAL
    endcase
    return_code = ShellExecute(0, 'open', pgm_name, args, start_dir,
                               shell_startup_flags)
    result = (return_code > 32)
    return(result)
  end StartPgm
#endif



#if EDITOR_VERSION < 4400h
  /*
    StrFind() 1.0

    If you have TSE Pro 4.0 or 4.2, then this proc implements the core of the
    built-in StrFind() function of TSE Pro 4.4.
    The StrFind() function searches a string or pattern inside another string
    and returns the position of the found string or zero.
    It works for strings like the regular Find() function does for files,
    so read the Help for the regular Find() function for the usage of the
    options, but apply these differences while reading:
    - Where the Find() (related) documentation refers to "file" and "line",
      StrFind() refers to "string".
    - The search option "g" ("global", meaning "from the start of the string")
      is implicit and can therefore always be omitted.
    As with the regular Find() function all characters are allowed as options,
    but here only these are acted upon: b, i, w, x, ^, $.

    Notable differences between the procedure below with TSE 4.4's built-in
    function:
    - The third parameter "options" is mandatory.
    - No fourth parameter "start" (actually occurrence: which one to search).
    - No fifth  parameter "len" (returning the length of the found text).

    Technical implementation notes:
    - To be reuseable elsewhere the procedure's source code is written to work
      independently of the rest of the source code.
      That said, it is intentionally not implemented as an include file, both
      for ease of installation and because one day another macro might need its
      omitted parameters, which would be an include file nightmare.
    - A tiny downside of the independent part is, that StrFind's buffer is not
      purged with the macro. To partially compensate for that if the macro is
      restarted, StrFind's possibly pre-existing buffer is searched for.
    - The fourth and fifth parameter are not implemented.
      - The first reason was that I estimated the tiny but actual performance
        gain and the easier function call to be more beneficial than the
        slight chance of a future use of these extra parameters.
      - The main reason turned out to be that in TSE 4.4 the fourth parameter
        "start" is erroneously documented and implemented.
        While this might be corrected in newer versions of TSE, it neither
        makes sense to me to faithfully reproduce these errors here, nor to
        make a correct implementation that will be replaced by an incorrect
        one if you upgrade to TSE 4.4.
  */
  integer strfind_id = 0
  integer proc StrFind(string needle, string haystack, string options)
    integer i                           = 0
    string  option                  [1] = ''
    integer org_id                      = GetBufferId()
    integer result                      = FALSE  // Zero.
    string  strfind_name [MAXSTRINGLEN] = ''
    string  validated_options       [7] = 'g'
    for i = 1 to Length(options)
      option = Lower(SubStr(options, i, 1))
      if      (option in 'b', 'i', 'w', 'x', '^', '$')
      and not Pos(option, validated_options)
        validated_options = validated_options + option
      endif
    endfor
    if strfind_id
      GotoBufferId(strfind_id)
      EmptyBuffer()
    else
      strfind_name = SplitPath(CurrMacroFilename(), _NAME_) + ':StrFind'
      strfind_id   = GetBufferId(strfind_name)
      if strfind_id
        GotoBufferId(strfind_id)
        EmptyBuffer()
      else
        strfind_id = CreateTempBuffer()
        ChangeCurrFilename(strfind_name, _DONT_PROMPT_|_DONT_EXPAND_|_OVERWRITE_)
      endif
    endif
    InsertText(haystack, _INSERT_)
    if lFind(needle, validated_options)
      result = CurrPos()
    endif
    GotoBufferId(org_id)
    return(result)
  end StrFind
#endif



#if EDITOR_VERSION < 4400h
  /*
    StrReplace() 1.0

    If you have TSE Pro 4.0 or 4.2, then this proc almost completely implements
    the built-in StrReplace() function of TSE Pro 4.4.
    The StrReplace() function replaces a string (pattern) inside a string.
    It works for strings like the Replace() function does for files, so read
    the Help for the Replace() function for the usage of the options, but apply
    these differences while reading:
    - Where Replace() refers to "file" and "line", StrReplace() refers to
      "string".
    - The options "g" ("global", meaning "from the start of the string")
      and "n" ("no questions", meaning "do not ask for confirmation on
      replacements") are implicitly always active, and can therefore be omitted.
    Notable differences between the procedure below with TSE 4.4's built-in
    function are, that here the fourth parameter "options" is mandatory
    and that the fifth parameter "start position" does not exist.
  */
  integer strreplace_id = 0
  string proc StrReplace(string needle, string haystack, string replacer, string options)
    integer i                      = 0
    integer org_id                 = GetBufferId()
    string  result  [MAXSTRINGLEN] = haystack
    string  validated_options [20] = 'gn'
    for i = 1 to Length(options)
      if (Lower(SubStr(options, i, 1)) in '0'..'9', 'b', 'i','w', 'x', '^', '$')
        validated_options = validated_options + SubStr(options, i, 1)
      endif
    endfor
    if strreplace_id == 0
      strreplace_id = CreateTempBuffer()
    else
      GotoBufferId(strreplace_id)
      EmptyBuffer()
    endif
    InsertText(haystack, _INSERT_)
    lReplace(needle, replacer, validated_options)
    result = GetText(1, CurrLineLen())
    GotoBufferId(org_id)
    return(result)
  end StrReplace
#endif



/*
  compare_versions()  v2.0

  This proc compares two version strings version1 and version2, and returns
  -1, 0 or 1 if version1 is smaller, equal, or greater than/to version2.

  For the comparison a version string is split into parts:
  - Explicitly by separating parts by a period.
  - Implicitly:
    - Any uninterrupted sequence of digits is a "number part".
    - Any uninterrupted sequence of other characters is a "string part".

  Spaces are mostly ignored. They are only significant:
  - Between two digits they signify that the digits belong to different parts.
  - Between two "other characters" they belong to the same string part.

  If the first version part is a single "v" or "V" then it is ignored.

  Two version strings are compared by comparing their respective version parts
  from left to right.

  Two number parts are compared numerically, e.g: "1" < "2" < "11" < "012".

  Any other combination of version parts is case-insensitively compared as
  strings, e.g: "12" < "one" < "three" < "two", or "a" < "B" < "c" < "d".

  Examples: See the included unit tests further on.

  v2.0
    Out in the wild there is an unversioned version of compare_versions(),
    that is more restricted in what version formats it can recognize,
    there here versioning of compare_versions() starts at v2.0.
*/

// compare_versions_standardize() is a helper proc for compare_versions().

string proc compare_versions_standardize(string p_version)
  integer char_nr                  = 0
  string  n_version [MAXSTRINGLEN] = Trim(p_version)

  // Replace any spaces between digits by one period. Needs two StrReplace()s.
  n_version = StrReplace('{[0-9]} #{[0-9]}', n_version, '\1.\2', 'x')
  n_version = StrReplace('{[0-9]} #{[0-9]}', n_version, '\1.\2', 'x')

  // Remove any spaces before and after digits.
  n_version = StrReplace(' #{[0-9]}', n_version, '\1', 'x')
  n_version = StrReplace('{[0-9]} #', n_version, '\1', 'x')

  // Remove any spaces before and after periods.
  n_version = StrReplace(' #{\.}', n_version, '\1', 'x')
  n_version = StrReplace('{\.} #', n_version, '\1', 'x')

  // Separate version parts by periods if they aren't yet.
  char_nr = 1
  while char_nr < Length(n_version)
    case n_version[char_nr:1]
      when '.'
        NoOp()
      when '0' .. '9'
        if not (n_version[char_nr+1:1] in '0' .. '9', '.')
          n_version = n_version[1:char_nr] + '.' + n_version[char_nr+1:MAXSTRINGLEN]
        endif
      otherwise
        if (n_version[char_nr+1:1] in '0' .. '9')
          n_version = n_version[1:char_nr] + '.' + n_version[char_nr+1:MAXSTRINGLEN]
        endif
    endcase
    char_nr = char_nr + 1
  endwhile
  // Remove a leading 'v' if it is by itself, i.e not part of a non-numeric string.
  if  (n_version[1:2] in 'v.', 'V.')
    n_version = n_version[3:MAXSTRINGLEN]
  endif
  return(n_version)
end compare_versions_standardize

integer proc compare_versions(string version1, string version2)
  integer result                 = 0
  string  v1_part [MAXSTRINGLEN] = ''
  string  v1_str  [MAXSTRINGLEN] = ''
  string  v2_part [MAXSTRINGLEN] = ''
  string  v2_str  [MAXSTRINGLEN] = ''
  integer v_num_parts            = 0
  integer v_part_nr              = 0
  v1_str      = compare_versions_standardize(version1)
  v2_str      = compare_versions_standardize(version2)
  v_num_parts = Max(NumTokens(v1_str, '.'), NumTokens(v2_str, '.'))
  repeat
    v_part_nr = v_part_nr + 1
    v1_part   = Trim(GetToken(v1_str, '.', v_part_nr))
    v2_part   = Trim(GetToken(v2_str, '.', v_part_nr))
    if  v1_part == ''
    and isDigit(v2_part)
      v1_part = '0'
    endif
    if v2_part == ''
    and isDigit(v1_part)
      v2_part = '0'
    endif
    if  isDigit(v1_part)
    and isDigit(v2_part)
      if     Val(v1_part) < Val(v2_part)
        result = -1
      elseif Val(v1_part) > Val(v2_part)
        result =  1
      endif
    else
      result = CmpiStr(v1_part, v2_part)
    endif
  until result    <> 0
     or v_part_nr >= v_num_parts
  return(result)
end compare_versions



/*
  wurn()  v1.0

  TSE's GUI version displays TSE's Warn() message in a possibly multi-line
  pop-up window, whereas TSE's console version would display it in the single
  status line.

  The wurn() procedure uses Warn() in TSE's GUI version and emulates the GUI's
  Warn() in TSE's console version, so the message looks the same across TSE
  versions.

  wurn() does not implement Warn()'s expression formatting,
  so use wurn(Format( ... )) if you need expression formatting.

  wurn() vs Warn():
  If  you don't need multi-line capability or long-line-wrapping capability,
  and you think the status line is just fine for TSE's console version,
  and you need expression formatting,
  then Warn() is the best option,
  otherwise wurn() or wurn(Format()) is a better choice.

  In the GUI version of TSE the Warn() command displays the formatted result
  string in this undocumented manner:
  - Not on the status line but in a pop-up window.
  - A carriage-return (CR, Chr(13)) character starts a new line.
  - A too long line is wrapped to a new line.
    Taking the pop-up window border and a separating space into account,
    the maximum displayable line length is the screen width minus 4.

  Examples:
    A GUI version of TSE with a screen width of 40 characters would display
      Warn('He sissed:', Chr(13), '':40:'s')
    and
      wurn(Format('He sissed:', Chr(13), '':40:'s'))
    in a pop-up windows as
      He sissed:
      ssssssssssssssssssssssssssssssssssss
      ssss

    A console version of TSE with a screen width of 40 characters would display
      wurn(Format('He sissed:', Chr(13), '':40:'s'))
    the same as the GUI version, but it would display
      Warn('He sissed:', Chr(13), '':40:'s')
    as the single status line:
      He sissed:Xssssssssssssss Press <Escape>
    where X is whatever your console version of TSE displays for a
    carriage-return character (possibly a music symbol)
    and where 26 "s" characters are cut off.
*/

integer wurn_id = 0

integer proc wurn(string warning_text)
  integer line_nr                       = 0
  integer line_parts                    = 0
  integer line_part_nr                  = 0
  string  line_part_text [MAXSTRINGLEN] = ''
  string  line_text      [MAXSTRINGLEN] = ''
  integer max_line_part_length          = 0
  integer org_id                        = 0
  if isGUI()
    Warn(warning_text)
  else
    if wurn_id
      EmptyBuffer(wurn_id)
    else
      org_id  = GetBufferId()
      wurn_id = CreateTempBuffer()
      GotoBufferId(org_id)
    endif
    if wurn_id
      max_line_part_length = Query(ScreenCols) - 4
      for line_nr = 1 to NumTokens(warning_text, Chr(13))
        line_text  = GetToken(warning_text, Chr(13), line_nr)
        line_parts = Length(line_text) / max_line_part_length + 1
        for line_part_nr = 1 to line_parts
          line_part_text = SubStr(line_text,
                                  (line_part_nr - 1) * max_line_part_length + 1,
                                  max_line_part_length)
          AddLine(line_part_text, wurn_id)
        endfor
      endfor
      MsgBoxBuff('Warning', _OK_, wurn_id)
    else
      Warn(warning_text)
    endif
  endif
  return(FALSE)
end wurn



// End of compatibility restrictions and mitigations.





// Global constants

string  CUAMARK_FIND_KEYS [37] = 'Ctrl F,Ctrl L,F3,CtrlShift L,Shift F3'

#define FIRST_OLDER_THAN_SECOND  -1
#define FIRST_EQUAL_TO_SECOND     0
#define FIRST_NEWER_THAN_SECOND   1

#define CCF_OPTIONS              _DONT_PROMPT_|_DONT_EXPAND_|_OVERWRITE_
#define MENU_VALUE_SIZE          32

string  MY_MACRO_VERSION   [5] = '1.1.1'
string  UPGRADE_URL       [42] = 'https://ecarlo.nl/tse/index.html#HiliteAll'



// Global variables

integer abort                                   = FALSE
string  cfg_duration                       [10] = ''
integer cfg_find_attr                           = 0
string  cfg_find_keys_section    [MAXSTRINGLEN] = ''
string  cfg_hiliting_after_replace          [7] = ''
integer cfg_replace_attr                        = 0
string  cfg_replace_keys_section [MAXSTRINGLEN] = ''
string  cfg_settings_section     [MAXSTRINGLEN] = ''
string  cfg_stop_keys_section    [MAXSTRINGLEN] = ''
string  cfg_version                        [13] = ''
integer do_replace_question_hiliting            = FALSE
string  find_options                       [23] = '^abcgilnvwx+1234567890$'
string  find_string              [MAXSTRINGLEN] = ''
integer hilite_all_finds                        = FALSE
integer hilite_all_replaces                     = FALSE
integer idle_timer                              = 0
integer menu_history_number                     = 0
string  my_macro_name            [MAXSTRINGLEN] = ''
integer purge_because_of_configuration          = FALSE
integer refresh_find_parameters                 = FALSE
integer refresh_replace_parameters              = FALSE
string  replace_options                    [23] = '^abcgilnvwx+1234567890$'
string  replace_string           [MAXSTRINGLEN] = ''
integer restart_menu                            = FALSE
string  saved_key_type                     [11] = '' // Intentionally 4 longer.
integer saved_menu_history_number               = 0
integer search_buffer_id                        = 0
integer search_results_buffer_id                = 0
integer stop_extra_hiliting                     = FALSE
integer sub_menu_history_number                 = FALSE



// General functions

proc to_beep_or_not_to_beep()
  if Query(Beep)
    Alarm()
  endif
end to_beep_or_not_to_beep

/*
  Return TSE's original LoadDir() if LoadDir() has been redirected
  by TSE's "-i" commandline option or a TSELOADDIR environment variable.
*/
string proc original_LoadDir()
  return(SplitPath(LoadDir(TRUE), _DRIVE_|_PATH_))
end original_LoadDir

/*
  If the text is longer than the new length, then is is cut off,
  otherwise it is lengthened by putting spaces around it.
*/
string proc center_text(string text, integer new_length)
  string centered_text [MAXSTRINGLEN] = ''
  if Length(text) >= new_length
    centered_text = SubStr(text, 1, new_length)
  else
    centered_text = Format('':((new_length - Length(text)) / 2), text)
    centered_text = Format(centered_text:(new_length * -1))
  endif
  return(centered_text)
end center_text

/*
  The following function determines the minimum amount of characters that a
  given regular expression can match. For the practical purpose that they are
  valid matches too, "^" and "$" are pretended to have length 1.
  The purpose of this procedure is, to be able to beforehand avoid searching
  for an empty string, which is logically pointless because it always succeeds
  (with one exception: past the end of the last line).
  Searching for an empty string in a loop makes it infinate and hangs a macro.
  Properly applied, using this procedure can avoid that.
*/
integer proc minimum_regexp_length(string s)
  integer addition           = 0
  integer i                  = 0
  integer NEXT_TIME          = 2
  integer orred_addition     = 0
  integer prev_addition      = 0
  integer prev_i             = 0
  integer result             = 0
  integer tag_level          = 0
  integer THIS_TIME          = 1
  integer use_orred_addition = FALSE
  // For each character.
  for i = 1 to Length(s)
    // Ignore this zero-length "character".
    if Lower(SubStr(s,i,2)) == "\c"
      i = i + 1
    else
      // Skip index for all these cases so they will be counted as one
      // character.
      case SubStr(s,i,1)
        when "["
          while i < Length(s)
          and   SubStr(s,i,1) <> "]"
            i = i + 1
          endwhile
        when "\"
          i = i + 1
          case Lower(SubStr(s,i,1))
            when "x"
              i = i + 2
            when "d", "o"
              i = i + 3
          endcase
      endcase
      // Now start counting.
      if use_orred_addition == NEXT_TIME
         use_orred_addition =  THIS_TIME
      endif
      // Count a literal character as one:
      if SubStr(s,i-1,1) == "\" // (Using the robustness of SubStr!)
        addition = 1
      // Count a tagged string as the length of its subexpression:
      elseif SubStr(s,i,1) == "{"
        prev_i = i
        tag_level = 1
        while i < Length(s)
        and   (tag_level <> 0 or SubStr(s,i,1) <> "}")
          i = i + 1
          if SubStr(s,i,1) == "{"
            tag_level = tag_level + 1
          elseif SubStr(s,i,1) == "}"
            tag_level = tag_level - 1
          endif
        endwhile
        addition = minimum_regexp_length(SubStr(s,prev_i+1,i-prev_i-1))
      // For a "previous character or tagged string may occur zero
      // times" operator: since it was already counted, subtract it.
      elseif SubStr(s,i,1) in "*", "@", "?"
        addition = -1 * Abs(prev_addition)
      // This is a tough one: the "or" operator.
      // For now subtract the length of the previous character or
      // tagged string, but remember doing so, because you might have
      // to add it again instead of the length of the character or
      // tagged string after the "or" operator.
      elseif SubStr(s,i,1) == "|"
        addition           = -1 * Abs(prev_addition)
        orred_addition     = Abs(prev_addition)
        use_orred_addition = NEXT_TIME
      else
      // Count ordinary characters as 1 character.
        addition = 1
      endif
      if use_orred_addition == THIS_TIME
        if orred_addition < addition
          addition = orred_addition
        endif
        use_orred_addition = FALSE
      endif
      result        = result + addition
      prev_addition = addition
    endif
  endfor
  return(result)
end minimum_regexp_length

string proc get_color_usages(integer color_attr)
  string result [MAXSTRINGLEN] = ''
  if color_attr == Query(TextAttr)
    result = result + ', Text'
  endif
  if color_attr == Query(HiliteAttr)
    result = result + ', Hilite'
  endif
  if color_attr == Query(BlockAttr)
    result = result + ', Block'
  endif
  if color_attr == Query(CursorAttr)
    result = result + ', Cursor'
  endif
  if color_attr == Query(CursorInBlockAttr)
    result = result + ', CursorInBlock'
  endif
  if color_attr == Query(Directive1Attr)
    result = result + ', Directive1'
  endif
  if color_attr == Query(Directive2Attr)
    result = result + ', Directive2'
  endif
  if color_attr == Query(Directive3Attr)
    result = result + ', Directive3'
  endif
  if color_attr == Query(MultiLnDlmt1Attr)
    result = result + ', MultiLnDlmt1'
  endif
  if color_attr == Query(MultiLnDlmt2Attr)
    result = result + ', MultiLnDlmt2'
  endif
  if color_attr == Query(MultiLnDlmt3Attr)
    result = result + ', MultiLnDlmt3'
  endif
  if color_attr == Query(SingleLnDlmt1Attr)
    result = result + ', SingleLnDlmt1'
  endif
  if color_attr == Query(SingleLnDlmt2Attr)
    result = result + ', SingleLnDlmt2'
  endif
  if color_attr == Query(SingleLnDlmt3Attr)
    result = result + ', SingleLnDlmt3'
  endif
  if color_attr == Query(ToEol1Attr)
    result = result + ', ToEol1'
  endif
  if color_attr == Query(ToEol2Attr)
    result = result + ', ToEol2'
  endif
  if color_attr == Query(ToEol3Attr)
    result = result + ', ToEol3'
  endif
  if color_attr == Query(Quote1Attr)
    result = result + ', Quote1'
  endif
  if color_attr == Query(Quote2Attr)
    result = result + ', Quote2'
  endif
  if color_attr == Query(Quote3Attr)
    result = result + ', Quote3'
  endif
  if color_attr == Query(NumberAttr)
    result = result + ', Number'
  endif
  if color_attr == Query(IncompleteQuoteAttr)
    result = result + ', IncompleteQuote'
  endif
  if color_attr == Query(Keywords1Attr)
    result = result + ', Keywords1'
  endif
  if color_attr == Query(Keywords2Attr)
    result = result + ', Keywords2'
  endif
  if color_attr == Query(Keywords3Attr)
    result = result + ', Keywords3'
  endif
  if color_attr == Query(Keywords4Attr)
    result = result + ', Keywords4'
  endif
  if color_attr == Query(Keywords5Attr)
    result = result + ', Keywords5'
  endif
  if color_attr == Query(Keywords6Attr)
    result = result + ', Keywords6'
  endif
  if color_attr == Query(Keywords7Attr)
    result = result + ', Keywords7'
  endif
  if color_attr == Query(Keywords8Attr)
    result = result + ', Keywords8'
  endif
  if color_attr == Query(Keywords9Attr)
    result = result + ', Keywords9'
  endif
  result = SubStr(result, 3, MAXSTRINGLEN)
  result = StrReplace(', ', result, ' and ', 'b1')
  return(result)
end get_color_usages

integer proc is_valid_text_color(integer color_attr)
  integer result           = TRUE
  integer background_color = color_attr  /  16
  integer foreground_color = color_attr mod 16
  if foreground_color == background_color
    result = FALSE
  endif
  return(result)
end is_valid_text_color

string proc get_color_name(integer color_attr)
  integer color_background = color_attr  /  16
  integer color_foreground = color_attr mod 16
  string  color_name  [33] = 'Intense Bright Magenta on Magenta'
  string  color_names [46] = 'Black Blue Green Cyan Red Magenta Yellow White'
  color_name = iif(color_foreground > 7, 'Bright ' , '')
             + GetToken(color_names, ' ', (color_foreground mod 8) + 1)
             + ' on '
             + iif(color_background > 7, 'Intense ', '')
             + GetToken(color_names, ' ', (color_background mod 8) + 1)
  return(color_name)
end get_color_name

integer proc select_color(integer param_color, string param_prompt)
  integer background                  = 0
  string  color_name             [33] = 'Intense Bright Magenta on Magenta'
  integer cols_per_color              = (Query(ScreenCols) - 2) / 16
  integer foreground                  = 0
  integer grid_color                  = 0
  integer left_margin                 = 0
  integer mouse_background            = 0
  integer mouse_foreground            = 0
  integer old_attr                    = Set(Attr, Color(bright white ON black))
  integer old_cursor                  = Set(Cursor, OFF)
  integer old_SpecialEffects          = 0
  integer prev_clockticks             = 0
  integer rows_per_color              = (Query(ScreenRows) - 3) / 16
  integer selected_color              = param_color
  integer selected_background         = 0
  integer selected_foreground         = 0
  integer sub_row                     = 0
  integer top_margin                  = (Query(ScreenRows) - 3 - rows_per_color * 16) / 2 + 1
  string  window_title [MAXSTRINGLEN] = ''
          left_margin                 = (Query(ScreenCols) - 2 - cols_per_color * 16) / 2 + 1
  BufferVideo()
  ClrScr()
  if param_color <> 0
     selected_foreground = param_color mod 16
     selected_background = param_color  /  16
  endif
  repeat
    color_name = get_color_name(selected_background * 16 + selected_foreground)
    if param_prompt == ''
      window_title = color_name
    else
      window_title = Format(param_prompt, color_name:36)
    endif
    PutStrAttrXY(1, 1, window_title, '', Color(bright white ON black))
    ClrEol()
    PopWinOpen(left_margin,
               top_margin,
               left_margin + (cols_per_color * 16) + 1,
               top_margin  + (rows_per_color * 16) + 1,
               1,
               '',
               Color(bright white ON black))
    for foreground = 0 to 15 // Rows.
      for background = 0 to 15 // Columns.
        for sub_row = 1 to rows_per_color
          if foreground <> 0 // Skip a bug in the PutStrAttrXY command
          or background <> 0 // up to and including TSE 4.2.
            grid_color = background * 16 + foreground
            PutStrAttrXY(background * cols_per_color + 1,
                         foreground * rows_per_color + sub_row,
                         Format((grid_color)
                                :cols_per_color:'0':16),
                         '',
                         grid_color)
          endif
        endfor
      endfor
    endfor
    old_SpecialEffects = Set(SpecialEffects,
                         Query(SpecialEffects) & ~ _DRAW_SHADOWS_)
    PopWinOpen(left_margin + selected_background * cols_per_color,
               top_margin  + selected_foreground * rows_per_color,
               left_margin + selected_background * cols_per_color + cols_per_color + 1,
               top_margin  + selected_foreground * rows_per_color + rows_per_color + 1,
               1,
               '',
               Color(bright white ON black))
    Set(SpecialEffects, old_SpecialEffects)
    UnBufferVideo()
    case GetKey()
      when <Home>, <GreyHome>
        selected_foreground = 0
        selected_background = 0
      when <end>, <GreyEnd>
        selected_foreground = 15
        selected_background = 15
      when <CursorDown>, <GreyCursorDown>
        if selected_foreground == 15
          selected_foreground = 0
        else
          selected_foreground = selected_foreground + 1
        endif
      when <CursorUp>, <GreyCursorUp>
        if selected_foreground == 0
          selected_foreground = 15
        else
          selected_foreground = selected_foreground - 1
        endif
      when <CursorRight>, <GreyCursorRight>
        if selected_background == 15
          selected_background = 0
        else
          selected_background = selected_background + 1
        endif
      when <CursorLeft>, <GreyCursorLeft>
        if selected_background == 0
          selected_background = 15
        else
          selected_background = selected_background - 1
        endif
      when <LeftBtn>
        /*
          The Delay(1) turned out to be necessary to give the system
          time to update the mouse status. An extra call to MouseStatus
          was useless, as could be expected, because TSE should already
          do that internally after a mousekey.
        */
        Delay(1)
        mouse_background = (Query(MouseX) - left_margin - 1) / cols_per_color
        mouse_foreground = (Query(MouseY) -  top_margin - 1) / rows_per_color
        if  (mouse_background in 0 .. 15)
        and (mouse_foreground in 0 .. 15)
          if  mouse_background == selected_background
          and mouse_foreground == selected_foreground
            if GetClockTicks() - prev_clockticks < 18
              PushKey(<Enter>)
            endif
          else
            selected_background = mouse_background
            selected_foreground = mouse_foreground
          endif
          prev_clockticks = GetClockTicks()
        endif
    endcase
    BufferVideo()
    PopWinClose()
    PopWinClose()
  until Query(Key) in <Enter>, <GreyEnter>, <Escape>
  UpdateDisplay(_ALL_WINDOWS_REFRESH_)
  UnBufferVideo()
  Set(Attr  , old_attr  )
  Set(Cursor, old_cursor)
  if Query(Key) <> <Escape>
    selected_color = selected_background * 16 + selected_foreground
  endif
  return(selected_color)
end select_color

proc write_profile_error()
  Warn('ERROR writing configuration to file "tse.ini"')
end write_profile_error

integer proc write_profile_int(string  section_name,
                               string  item_name,
                               integer item_value)
  integer result = WriteProfileInt(section_name,
                                   item_name,
                                   item_value)
  if not result
    write_profile_error()
  endif
  return(result)
end write_profile_int

integer proc write_profile_str(string section_name,
                               string item_name,
                               string item_value)
  integer result = WriteProfileStr(section_name,
                                   item_name,
                                   item_value)
  if not result
    write_profile_error()
  endif
  return(result)
end write_profile_str





// Code specific to this macro



string proc get_ui()
  integer org_id                = GetBufferId()
  integer tmp_id                = 0
  string  ui_fqn [MAXSTRINGLEN] = ''
  #if EDITOR_VERSION > 4400h
    // Pacify the compiler for not using these variables.
    org_id = org_id
    tmp_id = tmp_id
    // The following line will give a compiler error for TSE Beta versions
    // before v4.40.50 (17 Sep 2008). In the macro there is no cost-effective
    // way to solve that for an occurence that is highly unlikely to occur
    // and that has the more practical solution of upgrading.
    ui_fqn = Query(UIFilename)
  #else
    Wurn('I would like to search your user interface to find out which Find'
         + ' and Replace keys are configured, but your version of the editor'
         + ' cannot tell me which user interface it uses.' + Chr(13) + Chr(13)
         + 'Press OK and select which user interface file to search in.')
    tmp_id = EditFile(original_LoadDir() + 'ui' + SLASH + '*.ui')
    if tmp_id
      if CurrExt() == '.ui'
        ui_fqn = CurrFilename()
      endif
      GotoBufferId(org_id)
      AbandonFile (tmp_id)
    endif
  #endif
  return(ui_fqn)
end get_ui

integer proc add_auto_configurable_keys()
  integer changed_ui_open       = FALSE
  integer i                     = 0
  string  key_name         [30] = ''
  integer log_id                = 0
  integer ok                    = TRUE
  integer org_id                = GetBufferId()
  string  ui_fqn [MAXSTRINGLEN] = ''
  integer ui_id                 = 0
  log_id = CreateTempBuffer()
  if  isMacroLoaded        ('CUAMark')
  and EquiStr(GetProfileStr('CUAMark', 'UseCUAFindKeys', 'no'), 'yes')
    for i = 1 to NumTokens(CUAMARK_FIND_KEYS, ',')
      key_name = GetToken(CUAMARK_FIND_KEYS, ',', i)
      if not GetProfileInt(cfg_find_keys_section, key_name, FALSE)
        AddLine('Added Find key <' + key_name + '> from CUAMark.', log_id)
        if not write_profile_int(cfg_find_keys_section, key_name, TRUE)
          ok = FALSE
          break
        endif
      endif
    endfor
  endif
  if ok
    ui_fqn = get_ui()
    if  not FileExists(ui_fqn)
    and ui_fqn <> ''
      // This can be a valid exception: Just after a TSE installation TSE might
      // report a .ui file in an installation folder that no longer exists.
      // In that case we try the .ui file with the reported name but in the
      // standard "ui" folder.
      ui_fqn = original_LoadDir() + 'ui' + SLASH + SplitPath(ui_fqn, _NAME_|_EXT_)
    endif
  endif
  if  ok
  and FileExists(ui_fqn)
    ui_id = GetBufferId(ui_fqn)
    if ui_id
      GotoBufferId(ui_id)
      if FileChanged()
        changed_ui_open = TRUE
        ok = FALSE
        Warn('Cannot process open changed User Interface file.')
      else
        LoadBuffer(ui_fqn)
      endif
    else
      ui_id = CreateTempBuffer()
      LoadBuffer(ui_fqn)
    endif
    if ok
      BegFile()
      while lFind('<.#>[ \d009]#{Find}|{RepeatFind}(', 'ix')
        key_name = GetFoundText()
        if not lFind('<.*><.*>', 'cgx')
          key_name = StrReplace('<{.*}>.#', key_name, '\1', 'x')
          if not GetProfileInt(cfg_find_keys_section, key_name, FALSE)
            AddLine('Added Find key <' + key_name + '> from ' + ui_fqn,
                    log_id)
            if not write_profile_int(cfg_find_keys_section, key_name, TRUE)
              ok = FALSE
              break
            endif
          endif
        endif
        EndLine()
      endwhile
      BegFile()
      while lFind('<.#>[ \d009]#Replace(', 'ix')
        key_name = GetFoundText()
        if not lFind('<.*><.*>', 'cgx')
          key_name = StrReplace('<{.*}>.#', key_name, '\1', 'x')
          if not GetProfileInt(cfg_replace_keys_section, key_name, FALSE)
            AddLine('Added Replace key <' + key_name + '> from ' + ui_fqn,
                    log_id)
            if not write_profile_int(cfg_replace_keys_section, key_name, TRUE)
              ok = FALSE
              break
            endif
          endif
        endif
        EndLine()
      endwhile
    endif
  endif
  GotoBufferId(log_id)
  if NumLines()
    List('Added auto-configurable keys', LongestLineInBuffer())
  else
    Warn('No new keys were added.')
  endif
  restart_menu = TRUE
  GotoBufferId(org_id)
  AbandonFile(log_id)
  if      ui_id
  and not changed_ui_open
    AbandonFile(ui_id)
  endif
  return(ok)
end add_auto_configurable_keys




proc unhilite_existing_hiliting()
/*
  From the _idle_ hook, which is used to unhilite Finds, just
  an UpdateDisplay(_ALL_WINDOWS_REFRESH_) works perfectly,
  because it also restores the syntax hiliting.

  Alas, from the _nonedit_idle_ hook, which is used to unhilite during Replaces,
  an UpdateDisplay() also undoes Replace's own hiliting, so we can not use it.
  For further woe, we can only restore our own hiliting if it differs from the
  standard hiliting, because otherwise we can not recognize it.
  And to conclude our misery, even then we can only restore our hiliting to the
  default text colour, and not to its proper syntax hiliting.
*/
  string  s [MAXSTRINGLEN] = ''
  string  a [MAXSTRINGLEN] = ''
  integer x                = 0
  integer y                = 0
  integer i                = 0
  integer edit_state       = QueryEditState()
  if     edit_state == 0
    // During _idle_state_ event.
    UpdateDisplay(_ALL_WINDOWS_REFRESH_)
  elseif edit_state & STATE_EDITOR_PAUSED
    // This edit state occurs during a _nonedit_idle_ event
    // when Replace's per-occurrence (Yes/No/...) questions are asked.
    for y = Query(WindowY1) to Query(WindowY1) + Query(WindowRows) - 1
      x = Query(WindowX1)
      GetStrAttrXY(x, y, s, a, Query(WindowCols))
      for i = 1 to Length(a)
        if      (Asc(a[i]) in cfg_find_attr, cfg_replace_attr)
        and not (Asc(a[i]) in Query(HiliteAttr), Query(BlockAttr), Query(CursorInBlockAttr))
          a = a[1 .. (i - 1)] + Chr(Query(TextAttr)) + a[(i + 1) .. Length(a)]
        endif
      endfor
      PutStrAttrXY(x, y, s, a)
    endfor
  endif
end unhilite_existing_hiliting

proc after_getkey()
  string key_name [30] = KeyName(Query(Key))
  if GetProfileInt(cfg_find_keys_section, key_name, FALSE)
    hilite_all_finds        = TRUE
    refresh_find_parameters = TRUE
  endif
  if GetProfileInt(cfg_replace_keys_section, key_name, FALSE)
    hilite_all_replaces        = TRUE
    refresh_replace_parameters = TRUE
  endif
  if GetProfileInt(cfg_stop_keys_section, key_name, FALSE)
    stop_extra_hiliting        = TRUE
    hilite_all_finds           = FALSE
    hilite_all_replaces        = FALSE
    refresh_find_parameters    = FALSE
    refresh_replace_parameters = FALSE
  endif
  if hilite_all_replaces
    do_replace_question_hiliting = TRUE
  endif
end after_getkey

proc after_command()
  idle_timer = 0
  if hilite_all_replaces
    case cfg_hiliting_after_replace
      when 'Replace'
        find_string                = replace_string
        find_options               = replace_options
        hilite_all_finds           = TRUE
        hilite_all_replaces        = FALSE
        refresh_find_parameters    = FALSE
        refresh_replace_parameters = FALSE
      when 'Find'
        hilite_all_finds           = TRUE
        hilite_all_replaces        = FALSE
        refresh_find_parameters    = FALSE
        refresh_replace_parameters = FALSE
      otherwise
        stop_extra_hiliting        = TRUE
        hilite_all_finds           = FALSE
        hilite_all_replaces        = FALSE
        refresh_find_parameters    = FALSE
        refresh_replace_parameters = FALSE
    endcase
  endif
end after_command

proc hilite_edit_window(    integer search_options_history,
                        var integer refresh_search_parameters,
                        var string  search_string,
                        var string  search_options)
  string  a                     [1] = ''
  integer buffer_line_from          = 0
  integer buffer_line_to            = 0
  integer cfg_search_attr           = iif(search_options_history == _FIND_OPTIONS_HISTORY_, cfg_find_attr, cfg_replace_attr)
  integer marked_begin_col          = 0
  integer marked_length             = 0
  string  old_WordSet          [32] = ''
  string  option_extra_str      [1] = ''
  integer option_pos                = 0
  integer org_buffer_id             = GetBufferId()
  integer relative_curr_line        = 0
  integer result_line               = 0
  integer result_col                = 0
  integer result_length             = 0
  string  s                     [1] = ''
  integer x                         = 0
  integer y                         = 0
  // Sanity check 1
  if NumLines()
    // Which buffer lines are on the screen?
    buffer_line_from = CurrLine() - CurrRow() + 1
    buffer_line_to   = buffer_line_from + Query(WindowRows) - 1
    if buffer_line_to > NumLines()
      buffer_line_to = NumLines()
    endif
    relative_curr_line = CurrRow()
    // Sanity check 2   ()
    if  buffer_line_from <= buffer_line_to
    and (CurrLine() in buffer_line_from .. buffer_line_to)
      // Let's search the lines elsewhere and not disturb the current buffer.
      PushBlock()
      MarkLine(buffer_line_from, buffer_line_to)
      GotoBufferId(search_buffer_id)
      EmptyBuffer()
      // Sanity check 3   (A file with one empty line marks no block?)
      if CopyBlock()
        UnMarkBlock()
        if refresh_search_parameters
          search_string  =       GetHistoryStr(_FIND_HISTORY_, 1)
          search_options = Lower(GetHistoryStr(search_options_history, 1))
          refresh_search_parameters = FALSE
        endif
        GotoLine(relative_curr_line)
        // Remove unusable and not implemented options.
        repeat
          option_pos = StrFind('[0-9aglnv+]', search_options, 'x')
          if option_pos
            search_options =
              search_options[1                .. (option_pos - 1)      ] +
              search_options[(option_pos + 1) .. Length(search_options)]
          endif
        until not option_pos
        // Sanity check 4:
        //   Protect against a regular expression with an empty search result.
        if Pos('x', search_options)            == 0
        or minimum_regexp_length(search_string) > 0
          GotoBufferId(search_results_buffer_id)
          EmptyBuffer()
          GotoBufferId(search_buffer_id)
          option_extra_str = 'g'
          while lFind(search_string, search_options + option_extra_str)
            PushPosition()
            MarkFoundText()
            GotoBlockBeginCol()
            marked_begin_col = CurrCol()
            GotoBlockEndCol()
            marked_length = CurrCol() - marked_begin_col + 1
            UnMarkBlock()
            PopPosition()
            AddLine(Format(CurrLine(), ' ', CurrCol() , ' ', marked_length),
                    search_results_buffer_id)
            option_extra_str = '+'
          endwhile
          GotoBufferId(search_results_buffer_id)
          BegFile()
          if  NumLines()
          and CurrLineLen()
            old_WordSet = Set(WordSet, ChrSet('0-9'))
            repeat
              BegLine()
              result_line   = Val(GetWord())
              WordRight()
              result_col    = Val(GetWord())
              WordRight()
              result_length = Val(GetWord())
              GotoBufferId(org_buffer_id)
              // Note: CUAmark uses block marking instead of hiliting.
              x = Query(WindowX1) + result_col  - 1 - CurrXoffset()
              y = Query(WindowY1) + result_line - 1
              if      GetStrAttrXY(x, y, s, a, 1)
              and not (Asc(a) in Query(HiliteAttr), Query(BlockAttr),
                                 Query(CursorInBlockAttr))
                PutAttrXY(x, y, cfg_search_attr, result_length)
              endif
              GotoBufferId(search_results_buffer_id)
            until not Down()
            Set(WordSet, old_WordSet)
          endif
          if  search_options_history == _FIND_OPTIONS_HISTORY_
          and cfg_duration <> 'Perpetual'
            hilite_all_finds = FALSE
          endif
        endif
      endif
      GotoBufferId(org_buffer_id)
      PopBlock()
    endif
  endif
end hilite_edit_window

integer proc in_interactive_replace_question()
  string  a  [MAXSTRINGLEN] = ''
  integer screen_row_length = 0
  integer result            = FALSE
  string  s  [MAXSTRINGLEN] = ''
  integer status_line_row   = 0
  status_line_row   = iif(Query(StatusLineAtTop), iif(Query(ShowMainMenu), 2, 1), Query(ScreenRows))
  screen_row_length = GetStrAttrXY(1, status_line_row, s, a, MAXSTRINGLEN)
  if  screen_row_length
  and Asc(a[1]) == Query(MsgAttr)
  and Pos('    Replace (Yes/No/', s)
    result = TRUE
  endif
  return(result)
end in_interactive_replace_question

proc nonedit_idle()
  if hilite_all_replaces
    if NumLines()
      if do_replace_question_hiliting
        // Note, that a Replace passes at least three prompts,
        // and that a user might step out to their Help and footer functions,
        // so we wait for the interactive Replace questions.
        if in_interactive_replace_question()
          hilite_edit_window(_REPLACE_OPTIONS_HISTORY_,
                             refresh_replace_parameters,
                             replace_string,
                             replace_options)
        endif
        do_replace_question_hiliting = FALSE
      endif
    endif
  else
    if stop_extra_hiliting
      stop_extra_hiliting = FALSE
      unhilite_existing_hiliting()
    endif
  endif
end nonedit_idle

proc idle()
  if hilite_all_finds
    if NumLines()
      idle_timer = idle_timer - 1
      if idle_timer <= 0
        // Do not clog up the _idle_ hook.
        idle_timer = 36
        hilite_edit_window(_FIND_OPTIONS_HISTORY_,
                           refresh_find_parameters,
                           find_string,
                           find_options)
      endif
    endif
  else
    idle_timer = 0
    if stop_extra_hiliting
      stop_extra_hiliting = FALSE
      unhilite_existing_hiliting()
    endif
  endif
  if purge_because_of_configuration
    PurgeMacro(my_macro_name)
  endif
end idle

proc WhenPurged()
  AbandonFile(search_buffer_id)
  AbandonFile(search_results_buffer_id)
end WhenPurged

integer proc create_buffer(string buffer_specification, integer buffer_type)
  // Side effect: This proc switches to the created buffer!
  integer buffer_id                  = 0
  string  buffer_name [MAXSTRINGLEN] = my_macro_name + ':' + buffer_specification
  buffer_id = GetBufferId(buffer_name)
  if buffer_id
    GotoBufferId(buffer_id)
    EmptyBuffer()
  else
    buffer_id = CreateBuffer(buffer_name, buffer_type)
  endif
  return(buffer_id)
end create_buffer

integer proc upgrade_old_configuration()
  string  item_name  [MAXSTRINGLEN] = ''
  string  item_value [MAXSTRINGLEN] = ''
  integer ok                        = TRUE
  if compare_versions(cfg_version, '0.8') == FIRST_OLDER_THAN_SECOND
    if LoadProfileSection(my_macro_name + ':' + 'SearchKeys')
      while ok and GetNextProfileItem(item_name, item_value)
        ok = write_profile_str(my_macro_name + ':' + 'FindKeys',
                               item_name, item_value)
      endwhile
      if ok
        RemoveProfileSection(my_macro_name + ':' + 'SearchKeys')
      else
        Warn('Fatal error:'; my_macro_name; 'stopped.')
        PurgeMacro(my_macro_name)
      endif
    endif
  endif
  return(ok)
end upgrade_old_configuration

integer proc are_keys_defined(string cfg_keys_section)
  string item_name  [MAXSTRINGLEN] = ''
  string item_value [MAXSTRINGLEN] = ''
  LoadProfileSection(cfg_keys_section)
  return(GetNextProfileItem(item_name, item_value))
end are_keys_defined

proc WhenLoaded()
  integer org_buffer_id = GetBufferId()

  my_macro_name = SLASH   // Pacify the compiler for not using SLASH.
  my_macro_name = SplitPath(CurrMacroFilename(), _NAME_)

  #ifdef LINUX
    if compare_versions(VersionStr(), '4.41.35') == FIRST_OLDER_THAN_SECOND
      Warn('ERROR: In Linux the'; my_macro_name;
           'extension needs at least TSE v4.41.35.')
      PurgeMacro(my_macro_name)
      abort = TRUE
    endif
  #endif

  if not abort
    cfg_find_keys_section    = my_macro_name + ':' + 'FindKeys'
    cfg_replace_keys_section = my_macro_name + ':' + 'ReplaceKeys'
    cfg_stop_keys_section    = my_macro_name + ':' + 'StopKeys'
    cfg_settings_section     = my_macro_name + ':' + 'GeneralSettings'

    cfg_version                = GetProfileStr(cfg_settings_section,
                                               'Version' ,
                                               '0')
    case compare_versions(cfg_version, MY_MACRO_VERSION)
      when FIRST_OLDER_THAN_SECOND
        if upgrade_old_configuration()
          if write_profile_str(cfg_settings_section, 'Version', MY_MACRO_VERSION)
            cfg_version = MY_MACRO_VERSION
          else
            Warn('Fatal error:'; my_macro_name; 'stopped.')
            PurgeMacro(my_macro_name)
            abort = TRUE
          endif
        endif
      when FIRST_EQUAL_TO_SECOND
        NoOp()  // The normal case.
      when FIRST_NEWER_THAN_SECOND
        if YesNo('Older macro version detected. Configuration needs initialization!')
                  == 1
          RemoveProfileSection(cfg_find_keys_section   )
          RemoveProfileSection(cfg_replace_keys_section)
          RemoveProfileSection(cfg_stop_keys_section   )
          RemoveProfileSection(cfg_settings_section    )
          Warn('Execute the'; my_macro_name; 'macro to configure it.')
          PurgeMacro(my_macro_name)
          // No abort to give Main() a chance to fix the configuration.
        else
          Warn('Fatal error:'; my_macro_name; 'stopped.')
          PurgeMacro(my_macro_name)
          abort = TRUE
        endif
    endcase

    cfg_duration               = GetProfileStr(cfg_settings_section,
                                               'Duration',
                                               'Perpetual')
    cfg_hiliting_after_replace = GetProfileStr(cfg_settings_section,
                                               'HilitingAfterReplace',
                                               'Replace')
    cfg_find_attr              = GetProfileInt(cfg_settings_section,
                                               'FindColor',
                                               Query(HiliteAttr))
    cfg_replace_attr           = GetProfileInt(cfg_settings_section,
                                               'ReplaceColor',
                                               Query(HiliteAttr))

    search_buffer_id         = create_buffer('Search Buffer'        , _HIDDEN_)
    search_results_buffer_id = create_buffer('Search Results Buffer', _SYSTEM_)
    GotoBufferId(org_buffer_id)

    Hook(_AFTER_GETKEY_ , after_getkey )
    Hook(_AFTER_COMMAND_, after_command)
    Hook(_IDLE_         , idle         )
    Hook(_NONEDIT_IDLE_ , nonedit_idle )

    if  not are_keys_defined(cfg_find_keys_section)
    and not are_keys_defined(cfg_replace_keys_section)
      purge_because_of_configuration = TRUE
    endif
  endif
end WhenLoaded

proc show_help()
  string  full_macro_source_name [MAXSTRINGLEN] = SplitPath(CurrMacroFilename(), _DRIVE_|_PATH_|_NAME_) + '.s'
  string  help_file_name         [MAXSTRINGLEN] = '*** ' + my_macro_name + ' Help ***'
  integer hlp_id                                = GetBufferId(help_file_name)
  integer org_id                                = GetBufferId()
  integer tmp_id                                = 0
  if hlp_id
    GotoBufferId(hlp_id)
    UpdateDisplay()
  else
    tmp_id = CreateTempBuffer()
    if LoadBuffer(full_macro_source_name)
      // Separate / and * characters, otherwise my SynCase macro gets confused.
      if lFind('/' + '*', 'g')
        PushBlock()
        UnMarkBlock()
        Right(2)
        MarkChar()
        if not lFind('*' + '/', '')
          EndFile()
        endif
        MarkChar()
        CreateTempBuffer()
        CopyBlock()
        UnMarkBlock()
        PopBlock()
        BegFile()
        ChangeCurrFilename(help_file_name, CCF_OPTIONS)
        BufferType(_NORMAL_)
        FileChanged(FALSE)
        BrowseMode(TRUE)
        UpdateDisplay()
      else
        GotoBufferId(org_id)
        Wurn(Format('File "', full_macro_source_name,
                    '" has no multi-line comment block.'))
      endif
    else
      GotoBufferId(org_id)
      Wurn(Format('File "', full_macro_source_name, '" not found.'))
    endif
    AbandonFile(tmp_id)
  endif
end show_help

proc browse_website()
  #ifdef LINUX
    // Reportedly this is the most cross-Linux compatible way.
    // It worked out of the box for me: No Linux configuration was necessary.
    // Of course it will only work for Linux installations with a GUI.
    Dos('python -m webbrowser "' + UPGRADE_URL + '"',
        _DONT_PROMPT_|_DONT_CLEAR_|_RUN_DETACHED_)
  #else
    StartPgm(UPGRADE_URL, '', '', _DEFAULT_)
  #endif
end browse_website

integer proc get_stop_keys_flag()
  integer flag = _MF_SKIP_|_MF_GRAYED_
  if cfg_duration == 'Perpetual'
    if are_keys_defined(cfg_find_keys_section)
    or are_keys_defined(cfg_replace_keys_section)
      flag = _MF_CLOSE_ALL_BEFORE_|_MF_ENABLED_
    endif
  endif
  return(flag)
end get_stop_keys_flag

proc toggle_duration_setting()
  string new_cfg_duration [10] = iif(cfg_duration == 'Once', 'Perpetual', 'Once')
  if write_profile_str(cfg_settings_section, 'Duration', new_cfg_duration)
    cfg_duration = new_cfg_duration
    restart_menu = TRUE
  endif
end toggle_duration_setting

proc do_list_key(string list_key)
  integer key_code = 0
  case list_key
    when 'Del'
      KillLine()
    when 'Ins'
      Delay(5)
      Message('Press <Escape> or the new ', saved_key_type,
              ' key you want to add ... ')
      PopWinOpen(Query(ScreenCols) / 2 - 24,
                 Query(ScreenRows) / 2 -  1,
                 Query(ScreenCols) / 2 + 25,
                 Query(ScreenRows) / 2 +  1,
                 1,
                 '(or <Escape>)',
                 Color(intense black ON white))
      PutStrAttrXY(1, 1, center_text('Press your new '
                                     + saved_key_type
                                     + ' key now ...', 48),
                   '', Color(intense black ON white))
      key_code = GetKey()
      PopWinClose()
      Message('')
      if key_code <> <Escape>
        AddLine(KeyName(key_code))
        if lFind('^$', 'gx')
          KillLine()
        endif
      endif
  endcase
  PushKey(<Escape>)
end do_list_key

Keydef extra_list_keys
  <Ins>     do_list_key('Ins')
  <GreyIns> do_list_key('Ins')
  <KeyPad0> do_list_key('Ins')

  <Del>     do_list_key('Del')
  <GreyDel> do_list_key('Del')
  <KeyPad.> do_list_key('Del')
end extra_list_keys

proc extra_list_keys_starter()
   UnHook(extra_list_keys_starter)
   if Enable(extra_list_keys)
      ListFooter('{Ins}/{Del} Add or Delete a Key')
   endif
end extra_list_keys_starter

proc configure_keyboard_keys(string cfg_section_name)
  integer cfg_changed               = FALSE
  string  item_name  [MAXSTRINGLEN] = ''
  string  item_value [MAXSTRINGLEN] = ''
  string  key_name             [30] = ''
  integer lst_buffer_id             = 0
  integer ok                        = TRUE
  integer org_buffer_id             = GetBufferId()
  integer prev_FileChanged          = 0
  saved_key_type = GetToken(cfg_section_name, ':', 2)
  saved_key_type = SubStr(saved_key_type, 1, Length(saved_key_type) - 4)
  lst_buffer_id = CreateTempBuffer()
  if LoadProfileSection(cfg_section_name)
    while GetNextProfileItem(item_name, item_value)
      AddLine(item_name)
    endwhile
  endif
  repeat
    if NumLines()
      PushBlock()
      MarkLine(1, NumLines())
      Sort()
      UnMarkBlock()
      PopBlock()
    else
      AddLine('')
    endif
    prev_FileChanged = FileChanged()
    lFind(key_name, 'g^$')
    Hook(_LIST_STARTUP_, extra_list_keys_starter)
    List(cfg_section_name, Max(LongestLineInBuffer(), 44))
    key_name = GetText(1, CurrLineLen())
    Disable(extra_list_keys)
    cfg_changed = cfg_changed or FileChanged() <> prev_FileChanged
  until FileChanged() == prev_FileChanged
  if cfg_changed
    RemoveProfileSection(cfg_section_name)
    BegFile()
    repeat
      key_name = GetText(1, CurrLineLen())
      if key_name <> ''
        ok = write_profile_int(cfg_section_name, key_name, TRUE)
      endif
    until not ok
       or not Down()
  endif
  GotoBufferId(org_buffer_id)
  AbandonFile(lst_buffer_id)
  restart_menu = TRUE
end configure_keyboard_keys

string proc show_some_keys(string cfg_section_name)
  string  key_names  [MAXSTRINGLEN] = ''
  string  item_name            [30] = ''
  string  item_value            [1] = ''
  string  delimiter             [2] = ''
  integer last_comma_pos            = 0
  LoadProfileSection(cfg_section_name)
  while GetNextProfileItem(item_name, item_value)
    key_names = key_names + delimiter + item_name
    delimiter = ', '
  endwhile
  if Length(key_names) > MENU_VALUE_SIZE
    key_names      = key_names[1 : MENU_VALUE_SIZE]
    last_comma_pos = StrFind(',', key_names, 'b')
    key_names      = key_names[1 : last_comma_pos]
  endif
  return(key_names)
end show_some_keys

proc configure_color(string target)
  integer accept_selection   = FALSE
  integer stop_interrogation = FALSE
  integer new_attr           = 0
  integer old_attr           = iif(target == 'Find', cfg_find_attr, cfg_replace_attr)
  integer response           = _NONE_
  new_attr = old_attr
  repeat
    new_attr = select_color(new_attr, 'Pick colour for extra ' + target + 's')
    if is_valid_text_color(new_attr)
      if new_attr == old_attr
        stop_interrogation = TRUE
      elseif get_color_usages(new_attr) == ''
        accept_selection   = TRUE
        stop_interrogation = TRUE
      else
        response = MsgBox('Are you sure?',
                          'The selected '
                          + target
                          + ' colour "'
                          + get_color_name(new_attr)
                          + '" is also in use for '
                          + get_color_usages(new_attr)
                          + '.',
                         _YES_NO_CANCEL_)
        accept_selection   = (response == 1)
        stop_interrogation = (response <> 2)
      endif
    else
      Warn('For text you cannot pick the same foreground and background colour.')
    endif
  until stop_interrogation
  if     new_attr == old_attr
    Message('You did not change the colour.')
  elseif accept_selection
    if target == 'Find'
      cfg_find_attr    = new_attr
      WriteProfileInt(cfg_settings_section, 'FindColor'   , new_attr)
    else
      cfg_replace_attr = new_attr
      WriteProfileInt(cfg_settings_section, 'ReplaceColor', new_attr)
    endif
  endif
  restart_menu = TRUE
end configure_color

menu hiliting_after_replace_menu()
  title       = 'After a Replace is finished'
  x           = 5
  y           = 5
  history     = sub_menu_history_number

  'Keep hiliting the &Replace',,,
  "Keep hiliting with the Replace's search parameters"

  'Back to hiliting the last &Find',,,
  "Back to hiliting with the last Find's search parameters"

  '&No hiliting',,,
  'No hiliting'
end hiliting_after_replace_menu

proc configure_hiliting_after_replace()
  string old_cfg_hiliting_after_replace [7] = cfg_hiliting_after_replace
  saved_menu_history_number = MenuOption()
  sub_menu_history_number   = Pos(cfg_hiliting_after_replace,
                                  '       Replace Find    None')
                              / 8
  hiliting_after_replace_menu()
  case MenuOption()
    when 1
      cfg_hiliting_after_replace = 'Replace'
    when 2
      cfg_hiliting_after_replace = 'Find'
    when 3
      cfg_hiliting_after_replace = 'None'
  endcase
  if cfg_hiliting_after_replace <> old_cfg_hiliting_after_replace
    write_profile_str(cfg_settings_section, 'HilitingAfterReplace',
                      cfg_hiliting_after_replace)
  endif
  restart_menu = TRUE
end configure_hiliting_after_replace

string proc get_hiliting_after_replace_description()
  string description [MAXSTRINGLEN] = ''
  case cfg_hiliting_after_replace
    when 'Replace'
      description = 'Keep hiliting the Replace'
    when 'Find'
      description = 'Back to hiliting the last Find'
    when 'None'
      description = 'No hiliting'
  endcase
  return(description)
end get_hiliting_after_replace_description

integer proc get_key_dependency_flag(string cfg_keys_section)
  integer flag = _MF_SKIP_|_MF_GRAYED_
  if are_keys_defined(cfg_keys_section)
    flag = _MF_CLOSE_ALL_BEFORE_|_MF_ENABLED_
  endif
  return(flag)
end get_key_dependency_flag

integer proc get_duration_flag()
  integer flag = _MF_SKIP_|_MF_GRAYED_
  if are_keys_defined(cfg_find_keys_section)
  or are_keys_defined(cfg_replace_keys_section)
    flag = _MF_CLOSE_ALL_BEFORE_|_MF_ENABLED_
  endif
  return(flag)
end get_duration_flag

menu configuration_menu()
  title       = 'Hilite All - Configuration Menu'
  x           = 5
  y           = 5
  history     = menu_history_number

  '',, _MF_SKIP_

  '&Escape', NoOp(),, 'Exit this menu'
  '&Help ...', show_help(),, 'Read the documentation'

  '&Version ...'
    [cfg_version:MENU_VALUE_SIZE],
    browse_website(),,
    'Check the website for a newer version ...'

  '',, _MF_SKIP_
  'Assign side effects to existing keys',, _MF_DIVIDE_
  '',, _MF_SKIP_

  'Add auto-configurable Find and Replace &keys (limited) ...',
  add_auto_configurable_keys(),,
  'Limited capability to automatically add Find and Replace keys.'

  'Configure &Find keys ...'
    [show_some_keys(cfg_find_keys_section):MENU_VALUE_SIZE],
    configure_keyboard_keys(cfg_find_keys_section),,
    "Configure the existing Find keys for which to also hilite all matches."

  'Configure &Replace keys ...'
    [show_some_keys(cfg_replace_keys_section):MENU_VALUE_SIZE],
    configure_keyboard_keys(cfg_replace_keys_section),,
    "Configure the existing Replace keys for which to also hilite all matches."

  '',, _MF_SKIP_

  'Af&ter a Replace is finished'
    [get_hiliting_after_replace_description():MENU_VALUE_SIZE],
    configure_hiliting_after_replace(),
    get_key_dependency_flag(cfg_replace_keys_section),
    'Keep hiliting Replace, back to hiliting Find, or stop hiliting.'

  '',, _MF_SKIP_

  'Toggle the hiliting &duration'
    [cfg_duration:MENU_VALUE_SIZE],
    toggle_duration_setting(),
    get_duration_flag(),
    'After a search: Hilite then visible matches once, or keep hiliting new matches.'

  'Configure &Stop keys ...'
    [show_some_keys(cfg_stop_keys_section):MENU_VALUE_SIZE],
    configure_keyboard_keys(cfg_stop_keys_section),
    get_stop_keys_flag(),
    "Configure [existing] keys which should [also] stop extra hiliting."

  '',, _MF_SKIP_
  'Pick extra hiliting colours',, _MF_DIVIDE_
  '',, _MF_SKIP_

  'Colour for F&ind matches'
    [get_color_name(cfg_find_attr):MENU_VALUE_SIZE],
    configure_color('Find'),
    get_key_dependency_flag(cfg_find_keys_section),
    'Find hilites one match at a time. How to colour the other matches?'

  'Colour for Repl&ace matches'
    [get_color_name(cfg_replace_attr):MENU_VALUE_SIZE],
    configure_color('Replace'),
    get_key_dependency_flag(cfg_replace_keys_section),
    'Replace hilites one match at a time. How to colour the other matches?'
  '',, _MF_SKIP_
end configuration_menu

proc update_autoload_list()
  if are_keys_defined(cfg_find_keys_section)
  or are_keys_defined(cfg_replace_keys_section)
    if not isAutoLoaded()
      if not AddAutoLoadMacro(my_macro_name)
        to_beep_or_not_to_beep()
        Warn('Could not add macro "', my_macro_name, '" to file "tseload.dat".')
      endif
    endif
  else
    if isAutoLoaded()
      if not DelAutoLoadMacro(my_macro_name)
        to_beep_or_not_to_beep()
        Warn('Could not delete macro "', my_macro_name, '" from file "tseload.dat".')
      endif
    endif
  endif
end update_autoload_list

proc configure()
  integer button = 0
  Message('')
  repeat
    restart_menu = FALSE
    configuration_menu()
    menu_history_number = iif(saved_menu_history_number,
                              saved_menu_history_number, MenuOption())
    saved_menu_history_number = 0
    if  not restart_menu
    and     cfg_duration == 'Perpetual'
    and not are_keys_defined(cfg_stop_keys_section)
      button = MsgBoxEx('Configuration warning',
                'You configured "Perpetual" hiliting with no "Stop keys".'
                + Chr(13) + Chr(13) +
                'What to do?',
                '[&Auto-configure <Alt> as "Stop key"];[&Back to configuration];[&Cancel]')
      if     button == 1
        write_profile_int(cfg_stop_keys_section, 'Alt', TRUE)
      elseif button == 2
        restart_menu = TRUE
      endif
    endif
  until not restart_menu
  update_autoload_list()
end configure

proc Main()
  if not abort
    purge_because_of_configuration = FALSE
    configure()
    if not isAutoLoaded()
      PurgeMacro(my_macro_name)
    endif
  endif
end Main

