/**************************************************************************
  This macro is partially based on "Display Scheme", developed by George de
  Bruin in August, 1996, and heavily based on "Scheme", developed
  by Carlo Hogeveen in June 1999.

  12 Jun 2024, Carlo
  - No longer goes one line down after selecting a color scheme.
  - In the Color Schemes list, default sets the current line to the current
    color scheme (if TSE's current colors match a color scheme).
  - If no matching color scheme exists, then creates and goes to an empty line
    at the top.

  25 Jun 2024, Carlo
  - Fixed: Could not add a new color scheme from an empty line.
    (The first line is empty if no color scheme matches TSE's colors.)


 **************************************************************************/

/**************************************************************************
  Change the name of the specified scheme
 **************************************************************************/
integer proc rename_scheme(string old_scheme_name, string scheme_file)
    string new_scheme_name[255] = old_scheme_name

    if Ask('Rename "' + old_scheme_name + '" to :', new_scheme_name) and new_scheme_name <> ""
        RenameProfileSection(old_scheme_name, new_scheme_name, scheme_file)
        return (TRUE)
    endif
    return (FALSE)
end

/**************************************************************************
  Update the colors of the specified scheme with the current
  editor colors.
 **************************************************************************/
proc update_scheme(string scheme_name, string scheme_file)
    WriteProfileInt(scheme_name, "CurrWinBorderAttr"  , Query(CurrWinBorderAttr)  , scheme_file)
    WriteProfileInt(scheme_name, "OtherWinBorderAttr" , Query(OtherWinBorderAttr) , scheme_file)
    WriteProfileInt(scheme_name, "MenuTextAttr"       , Query(MenuTextAttr)       , scheme_file)
    WriteProfileInt(scheme_name, "MenuTextLtrAttr"    , Query(MenuTextLtrAttr)    , scheme_file)
    WriteProfileInt(scheme_name, "MenuGrayAttr"       , Query(MenuGrayAttr)       , scheme_file)
    WriteProfileInt(scheme_name, "MenuSelectGrayAttr" , Query(MenuSelectGrayAttr) , scheme_file)
    WriteProfileInt(scheme_name, "MenuBorderAttr"     , Query(MenuBorderAttr)     , scheme_file)
    WriteProfileInt(scheme_name, "MenuSelectAttr"     , Query(MenuSelectAttr)     , scheme_file)
    WriteProfileInt(scheme_name, "MenuSelectLtrAttr"  , Query(MenuSelectLtrAttr)  , scheme_file)
    WriteProfileInt(scheme_name, "BlockAttr"          , Query(BlockAttr)          , scheme_file)
    WriteProfileInt(scheme_name, "CursorInBlockAttr"  , Query(CursorInBlockAttr)  , scheme_file)
    WriteProfileInt(scheme_name, "CursorAttr"         , Query(CursorAttr)         , scheme_file)
    WriteProfileInt(scheme_name, "HiLiteAttr"         , Query(HiliteAttr)         , scheme_file)
    WriteProfileInt(scheme_name, "TextAttr"           , Query(TextAttr)           , scheme_file)
    WriteProfileInt(scheme_name, "EOFMarkerAttr"      , Query(EofMarkerAttr)      , scheme_file)
    WriteProfileInt(scheme_name, "MsgAttr"            , Query(MsgAttr)            , scheme_file)
    WriteProfileInt(scheme_name, "StatusLineAttr"     , Query(StatusLineAttr)     , scheme_file)
    WriteProfileInt(scheme_name, "HelpInfoAttr"       , Query(HelpInfoAttr)       , scheme_file)
    WriteProfileInt(scheme_name, "HelpLinkAttr"       , Query(HelpLinkAttr)       , scheme_file)
    WriteProfileInt(scheme_name, "HelpSelectAttr"     , Query(HelpSelectAttr)     , scheme_file)
    WriteProfileInt(scheme_name, "HelpBoldAttr"       , Query(HelpBoldAttr)       , scheme_file)
    WriteProfileInt(scheme_name, "HelpItalicsAttr"    , Query(HelpItalicsAttr)    , scheme_file)
    WriteProfileInt(scheme_name, "HelpTextAttr"       , Query(HelpTextAttr)       , scheme_file)
    WriteProfileInt(scheme_name, "IncompleteQuoteAttr", Query(IncompleteQuoteAttr), scheme_file)
    WriteProfileInt(scheme_name, "NumberAttr"         , Query(NumberAttr)         , scheme_file)
    WriteProfileInt(scheme_name, "Directive1Attr"     , Query(Directive1Attr)     , scheme_file)
    WriteProfileInt(scheme_name, "Directive2Attr"     , Query(Directive2Attr)     , scheme_file)
    WriteProfileInt(scheme_name, "Directive3Attr"     , Query(Directive3Attr)     , scheme_file)
    WriteProfileInt(scheme_name, "MultiLnDlmt1Attr"   , Query(MultiLnDlmt1Attr)   , scheme_file)
    WriteProfileInt(scheme_name, "MultiLnDlmt2Attr"   , Query(MultiLnDlmt2Attr)   , scheme_file)
    WriteProfileInt(scheme_name, "MultiLnDlmt3Attr"   , Query(MultiLnDlmt3Attr)   , scheme_file)
    WriteProfileInt(scheme_name, "SingleLnDlmt1Attr"  , Query(SingleLnDlmt1Attr)  , scheme_file)
    WriteProfileInt(scheme_name, "SingleLnDlmt2Attr"  , Query(SingleLnDlmt2Attr)  , scheme_file)
    WriteProfileInt(scheme_name, "SingleLnDlmt3Attr"  , Query(SingleLnDlmt3Attr)  , scheme_file)
    WriteProfileInt(scheme_name, "Quote1Attr"         , Query(Quote1Attr)         , scheme_file)
    WriteProfileInt(scheme_name, "Quote2Attr"         , Query(Quote2Attr)         , scheme_file)
    WriteProfileInt(scheme_name, "Quote3Attr"         , Query(Quote3Attr)         , scheme_file)
    WriteProfileInt(scheme_name, "ToEOL1Attr"         , Query(ToEol1Attr)         , scheme_file)
    WriteProfileInt(scheme_name, "ToEOL2Attr"         , Query(ToEol2Attr)         , scheme_file)
    WriteProfileInt(scheme_name, "ToEOL3Attr"         , Query(ToEol3Attr)         , scheme_file)
    WriteProfileInt(scheme_name, "KeyWords1Attr"      , Query(Keywords1Attr)      , scheme_file)
    WriteProfileInt(scheme_name, "KeyWords2Attr"      , Query(Keywords2Attr)      , scheme_file)
    WriteProfileInt(scheme_name, "KeyWords3Attr"      , Query(Keywords3Attr)      , scheme_file)
    WriteProfileInt(scheme_name, "KeyWords4Attr"      , Query(Keywords4Attr)      , scheme_file)
    WriteProfileInt(scheme_name, "KeyWords5Attr"      , Query(Keywords5Attr)      , scheme_file)
    WriteProfileInt(scheme_name, "KeyWords6Attr"      , Query(Keywords6Attr)      , scheme_file)
    WriteProfileInt(scheme_name, "KeyWords7Attr"      , Query(Keywords7Attr)      , scheme_file)
    WriteProfileInt(scheme_name, "KeyWords8Attr"      , Query(Keywords8Attr)      , scheme_file)
    WriteProfileInt(scheme_name, "KeyWords9Attr"      , Query(Keywords9Attr)      , scheme_file)
end

/**************************************************************************
  Load the specified scheme
 **************************************************************************/
proc load_scheme(string scheme_name, string scheme_file)
    Set(CurrWinBorderAttr  , GetProfileInt(scheme_name, "CurrWinBorderAttr"  , Query(CurrWinBorderAttr)  , scheme_file))
    Set(OtherWinBorderAttr , GetProfileInt(scheme_name, "OtherWinBorderAttr" , Query(OtherWinBorderAttr) , scheme_file))
    Set(MenuTextAttr       , GetProfileInt(scheme_name, "MenuTextAttr"       , Query(MenuTextAttr)       , scheme_file))
    Set(MenuTextLtrAttr    , GetProfileInt(scheme_name, "MenuTextLtrAttr"    , Query(MenuTextLtrAttr)    , scheme_file))
    Set(MenuGrayAttr       , GetProfileInt(scheme_name, "MenuGrayAttr"       , Query(MenuGrayAttr)       , scheme_file))
    Set(MenuSelectGrayAttr , GetProfileInt(scheme_name, "MenuSelectGrayAttr" , Query(MenuSelectGrayAttr) , scheme_file))
    Set(MenuBorderAttr     , GetProfileInt(scheme_name, "MenuBorderAttr"     , Query(MenuBorderAttr)     , scheme_file))
    Set(MenuSelectAttr     , GetProfileInt(scheme_name, "MenuSelectAttr"     , Query(MenuSelectAttr)     , scheme_file))
    Set(MenuSelectLtrAttr  , GetProfileInt(scheme_name, "MenuSelectLtrAttr"  , Query(MenuSelectLtrAttr)  , scheme_file))
    Set(BlockAttr          , GetProfileInt(scheme_name, "BlockAttr"          , Query(BlockAttr)          , scheme_file))
    Set(CursorInBlockAttr  , GetProfileInt(scheme_name, "CursorInBlockAttr"  , Query(CursorInBlockAttr)  , scheme_file))
    Set(CursorAttr         , GetProfileInt(scheme_name, "CursorAttr"         , Query(CursorAttr)         , scheme_file))
    Set(HiliteAttr         , GetProfileInt(scheme_name, "HiLiteAttr"         , Query(HiliteAttr)         , scheme_file))
    Set(TextAttr           , GetProfileInt(scheme_name, "TextAttr"           , Query(TextAttr)           , scheme_file))
    Set(EofMarkerAttr      , GetProfileInt(scheme_name, "EOFMarkerAttr"      , Query(EofMarkerAttr)      , scheme_file))
    Set(MsgAttr            , GetProfileInt(scheme_name, "MsgAttr"            , Query(MsgAttr)            , scheme_file))
    Set(StatusLineAttr     , GetProfileInt(scheme_name, "StatusLineAttr"     , Query(StatusLineAttr)     , scheme_file))
    Set(HelpInfoAttr       , GetProfileInt(scheme_name, "HelpInfoAttr"       , Query(HelpInfoAttr)       , scheme_file))
    Set(HelpLinkAttr       , GetProfileInt(scheme_name, "HelpLinkAttr"       , Query(HelpLinkAttr)       , scheme_file))
    Set(HelpSelectAttr     , GetProfileInt(scheme_name, "HelpSelectAttr"     , Query(HelpSelectAttr)     , scheme_file))
    Set(HelpBoldAttr       , GetProfileInt(scheme_name, "HelpBoldAttr"       , Query(HelpBoldAttr)       , scheme_file))
    Set(HelpItalicsAttr    , GetProfileInt(scheme_name, "HelpItalicsAttr"    , Query(HelpItalicsAttr)    , scheme_file))
    Set(HelpTextAttr       , GetProfileInt(scheme_name, "HelpTextAttr"       , Query(HelpTextAttr)       , scheme_file))
    Set(IncompleteQuoteAttr, GetProfileInt(scheme_name, "IncompleteQuoteAttr", Query(IncompleteQuoteAttr), scheme_file))
    Set(NumberAttr         , GetProfileInt(scheme_name, "NumberAttr"         , Query(NumberAttr)         , scheme_file))
    Set(Directive1Attr     , GetProfileInt(scheme_name, "Directive1Attr"     , Query(Directive1Attr)     , scheme_file))
    Set(Directive2Attr     , GetProfileInt(scheme_name, "Directive2Attr"     , Query(Directive2Attr)     , scheme_file))
    Set(Directive3Attr     , GetProfileInt(scheme_name, "Directive3Attr"     , Query(Directive3Attr)     , scheme_file))
    Set(MultiLnDlmt1Attr   , GetProfileInt(scheme_name, "MultiLnDlmt1Attr"   , Query(MultiLnDlmt1Attr)   , scheme_file))
    Set(MultiLnDlmt2Attr   , GetProfileInt(scheme_name, "MultiLnDlmt2Attr"   , Query(MultiLnDlmt2Attr)   , scheme_file))
    Set(MultiLnDlmt3Attr   , GetProfileInt(scheme_name, "MultiLnDlmt3Attr"   , Query(MultiLnDlmt3Attr)   , scheme_file))
    Set(SingleLnDlmt1Attr  , GetProfileInt(scheme_name, "SingleLnDlmt1Attr"  , Query(SingleLnDlmt1Attr)  , scheme_file))
    Set(SingleLnDlmt2Attr  , GetProfileInt(scheme_name, "SingleLnDlmt2Attr"  , Query(SingleLnDlmt2Attr)  , scheme_file))
    Set(SingleLnDlmt3Attr  , GetProfileInt(scheme_name, "SingleLnDlmt3Attr"  , Query(SingleLnDlmt3Attr)  , scheme_file))
    Set(Quote1Attr         , GetProfileInt(scheme_name, "Quote1Attr"         , Query(Quote1Attr)         , scheme_file))
    Set(Quote2Attr         , GetProfileInt(scheme_name, "Quote2Attr"         , Query(Quote2Attr)         , scheme_file))
    Set(Quote3Attr         , GetProfileInt(scheme_name, "Quote3Attr"         , Query(Quote3Attr)         , scheme_file))
    Set(ToEol1Attr         , GetProfileInt(scheme_name, "ToEOL1Attr"         , Query(ToEol1Attr)         , scheme_file))
    Set(ToEol2Attr         , GetProfileInt(scheme_name, "ToEOL2Attr"         , Query(ToEol2Attr)         , scheme_file))
    Set(ToEol3Attr         , GetProfileInt(scheme_name, "ToEOL3Attr"         , Query(ToEol3Attr)         , scheme_file))
    Set(Keywords1Attr      , GetProfileInt(scheme_name, "KeyWords1Attr"      , Query(Keywords1Attr)      , scheme_file))
    Set(Keywords2Attr      , GetProfileInt(scheme_name, "KeyWords2Attr"      , Query(Keywords2Attr)      , scheme_file))
    Set(Keywords3Attr      , GetProfileInt(scheme_name, "KeyWords3Attr"      , Query(Keywords3Attr)      , scheme_file))
    Set(Keywords4Attr      , GetProfileInt(scheme_name, "KeyWords4Attr"      , Query(Keywords4Attr)      , scheme_file))
    Set(Keywords5Attr      , GetProfileInt(scheme_name, "KeyWords5Attr"      , Query(Keywords5Attr)      , scheme_file))
    Set(Keywords6Attr      , GetProfileInt(scheme_name, "KeyWords6Attr"      , Query(Keywords6Attr)      , scheme_file))
    Set(Keywords7Attr      , GetProfileInt(scheme_name, "KeyWords7Attr"      , Query(Keywords7Attr)      , scheme_file))
    Set(Keywords8Attr      , GetProfileInt(scheme_name, "KeyWords8Attr"      , Query(Keywords8Attr)      , scheme_file))
    Set(Keywords9Attr      , GetProfileInt(scheme_name, "KeyWords9Attr"      , Query(Keywords9Attr)      , scheme_file))
end

/**************************************************************************
  Add a new scheme
 **************************************************************************/
integer proc add_scheme(string scheme_file)
    string scheme_name[255] = ""

    if Ask("Save current color scheme as:", scheme_name) and scheme_name <> ""
        update_scheme(scheme_name, scheme_file)
        return (TRUE)
   endif
   return (FALSE)
end

/*
  Pre-condition: A newly created color scheme buffer is current.

  This proc either sets the current line to the first listed color scheme that
  matches TSE's current color, or it creates an empty line at the start and
  makes that the current line.
*/
proc set_currline_to_color_scheme(string scheme_file)
  integer found                      = FALSE
  string  scheme_name [MAXSTRINGLEN] = ''

  BegFile()
  repeat
    scheme_name = GetText(1, MAXSTRINGLEN)

    if  GetProfileInt(scheme_name, "CurrWinBorderAttr"  , Query(CurrWinBorderAttr)  , scheme_file) == Query(CurrWinBorderAttr)
    and GetProfileInt(scheme_name, "OtherWinBorderAttr" , Query(OtherWinBorderAttr) , scheme_file) == Query(OtherWinBorderAttr)
    and GetProfileInt(scheme_name, "MenuTextAttr"       , Query(MenuTextAttr)       , scheme_file) == Query(MenuTextAttr)
    and GetProfileInt(scheme_name, "MenuTextLtrAttr"    , Query(MenuTextLtrAttr)    , scheme_file) == Query(MenuTextLtrAttr)
    and GetProfileInt(scheme_name, "MenuGrayAttr"       , Query(MenuGrayAttr)       , scheme_file) == Query(MenuGrayAttr)
    and GetProfileInt(scheme_name, "MenuSelectGrayAttr" , Query(MenuSelectGrayAttr) , scheme_file) == Query(MenuSelectGrayAttr)
    and GetProfileInt(scheme_name, "MenuBorderAttr"     , Query(MenuBorderAttr)     , scheme_file) == Query(MenuBorderAttr)
    and GetProfileInt(scheme_name, "MenuSelectAttr"     , Query(MenuSelectAttr)     , scheme_file) == Query(MenuSelectAttr)
    and GetProfileInt(scheme_name, "MenuSelectLtrAttr"  , Query(MenuSelectLtrAttr)  , scheme_file) == Query(MenuSelectLtrAttr)
    and GetProfileInt(scheme_name, "BlockAttr"          , Query(BlockAttr)          , scheme_file) == Query(BlockAttr)
    and GetProfileInt(scheme_name, "CursorInBlockAttr"  , Query(CursorInBlockAttr)  , scheme_file) == Query(CursorInBlockAttr)
    and GetProfileInt(scheme_name, "CursorAttr"         , Query(CursorAttr)         , scheme_file) == Query(CursorAttr)
    and GetProfileInt(scheme_name, "HiLiteAttr"         , Query(HiliteAttr)         , scheme_file) == Query(HiliteAttr)
    and GetProfileInt(scheme_name, "TextAttr"           , Query(TextAttr)           , scheme_file) == Query(TextAttr)
    and GetProfileInt(scheme_name, "EOFMarkerAttr"      , Query(EofMarkerAttr)      , scheme_file) == Query(EofMarkerAttr)
    and GetProfileInt(scheme_name, "MsgAttr"            , Query(MsgAttr)            , scheme_file) == Query(MsgAttr)
    and GetProfileInt(scheme_name, "StatusLineAttr"     , Query(StatusLineAttr)     , scheme_file) == Query(StatusLineAttr)
    and GetProfileInt(scheme_name, "HelpInfoAttr"       , Query(HelpInfoAttr)       , scheme_file) == Query(HelpInfoAttr)
    and GetProfileInt(scheme_name, "HelpLinkAttr"       , Query(HelpLinkAttr)       , scheme_file) == Query(HelpLinkAttr)
    and GetProfileInt(scheme_name, "HelpSelectAttr"     , Query(HelpSelectAttr)     , scheme_file) == Query(HelpSelectAttr)
    and GetProfileInt(scheme_name, "HelpBoldAttr"       , Query(HelpBoldAttr)       , scheme_file) == Query(HelpBoldAttr)
    and GetProfileInt(scheme_name, "HelpItalicsAttr"    , Query(HelpItalicsAttr)    , scheme_file) == Query(HelpItalicsAttr)
    and GetProfileInt(scheme_name, "HelpTextAttr"       , Query(HelpTextAttr)       , scheme_file) == Query(HelpTextAttr)
    and GetProfileInt(scheme_name, "IncompleteQuoteAttr", Query(IncompleteQuoteAttr), scheme_file) == Query(IncompleteQuoteAttr)
    and GetProfileInt(scheme_name, "NumberAttr"         , Query(NumberAttr)         , scheme_file) == Query(NumberAttr)
    and GetProfileInt(scheme_name, "Directive1Attr"     , Query(Directive1Attr)     , scheme_file) == Query(Directive1Attr)
    and GetProfileInt(scheme_name, "Directive2Attr"     , Query(Directive2Attr)     , scheme_file) == Query(Directive2Attr)
    and GetProfileInt(scheme_name, "Directive3Attr"     , Query(Directive3Attr)     , scheme_file) == Query(Directive3Attr)
    and GetProfileInt(scheme_name, "MultiLnDlmt1Attr"   , Query(MultiLnDlmt1Attr)   , scheme_file) == Query(MultiLnDlmt1Attr)
    and GetProfileInt(scheme_name, "MultiLnDlmt2Attr"   , Query(MultiLnDlmt2Attr)   , scheme_file) == Query(MultiLnDlmt2Attr)
    and GetProfileInt(scheme_name, "MultiLnDlmt3Attr"   , Query(MultiLnDlmt3Attr)   , scheme_file) == Query(MultiLnDlmt3Attr)
    and GetProfileInt(scheme_name, "SingleLnDlmt1Attr"  , Query(SingleLnDlmt1Attr)  , scheme_file) == Query(SingleLnDlmt1Attr)
    and GetProfileInt(scheme_name, "SingleLnDlmt2Attr"  , Query(SingleLnDlmt2Attr)  , scheme_file) == Query(SingleLnDlmt2Attr)
    and GetProfileInt(scheme_name, "SingleLnDlmt3Attr"  , Query(SingleLnDlmt3Attr)  , scheme_file) == Query(SingleLnDlmt3Attr)
    and GetProfileInt(scheme_name, "Quote1Attr"         , Query(Quote1Attr)         , scheme_file) == Query(Quote1Attr)
    and GetProfileInt(scheme_name, "Quote2Attr"         , Query(Quote2Attr)         , scheme_file) == Query(Quote2Attr)
    and GetProfileInt(scheme_name, "Quote3Attr"         , Query(Quote3Attr)         , scheme_file) == Query(Quote3Attr)
    and GetProfileInt(scheme_name, "ToEOL1Attr"         , Query(ToEol1Attr)         , scheme_file) == Query(ToEol1Attr)
    and GetProfileInt(scheme_name, "ToEOL2Attr"         , Query(ToEol2Attr)         , scheme_file) == Query(ToEol2Attr)
    and GetProfileInt(scheme_name, "ToEOL3Attr"         , Query(ToEol3Attr)         , scheme_file) == Query(ToEol3Attr)
    and GetProfileInt(scheme_name, "KeyWords1Attr"      , Query(Keywords1Attr)      , scheme_file) == Query(Keywords1Attr)
    and GetProfileInt(scheme_name, "KeyWords2Attr"      , Query(Keywords2Attr)      , scheme_file) == Query(Keywords2Attr)
    and GetProfileInt(scheme_name, "KeyWords3Attr"      , Query(Keywords3Attr)      , scheme_file) == Query(Keywords3Attr)
    and GetProfileInt(scheme_name, "KeyWords4Attr"      , Query(Keywords4Attr)      , scheme_file) == Query(Keywords4Attr)
    and GetProfileInt(scheme_name, "KeyWords5Attr"      , Query(Keywords5Attr)      , scheme_file) == Query(Keywords5Attr)
    and GetProfileInt(scheme_name, "KeyWords6Attr"      , Query(Keywords6Attr)      , scheme_file) == Query(Keywords6Attr)
    and GetProfileInt(scheme_name, "KeyWords7Attr"      , Query(Keywords7Attr)      , scheme_file) == Query(Keywords7Attr)
    and GetProfileInt(scheme_name, "KeyWords8Attr"      , Query(Keywords8Attr)      , scheme_file) == Query(Keywords8Attr)
    and GetProfileInt(scheme_name, "KeyWords9Attr"      , Query(Keywords9Attr)      , scheme_file) == Query(Keywords9Attr)
      found = TRUE
    endif
  until found
     or not Down()

  if not found
    BegFile()
    InsertLine()
  endif
end set_currline_to_color_scheme


/**************************************************************************
  Get a (sorted) list of the scheme section names into an editor buffer
 **************************************************************************/
proc build_scheme_buffer(var integer scheme_buffer, string scheme_file)
   integer msg_level
   string scheme_name[255] = ""

   PushPosition()

   if scheme_buffer == 0
      scheme_buffer = CreateTempBuffer()
   else
      GotoBufferId(scheme_buffer)
      EmptyBuffer()
   endif

   if LoadProfileSectionNames(scheme_file)
      while GetNextProfileSectionName(scheme_name)
         AddLine(scheme_name, scheme_buffer)
      endwhile
   endif

   PushBlock()
   MarkLine(1, NumLines())
   msg_level = Set(MsgLevel, _WARNINGS_ONLY_)
   Sort(_IGNORE_CASE_)
   Set(MsgLevel, msg_level)
   PopBlock()

   set_currline_to_color_scheme(scheme_file)

   PopPosition()
end

constant kADD = -1, kUPDATE = -2, kRENAME = -3, kDELETE = -4, kPalette = -5

Keydef list_keys
<CursorUp>    if not Up() EndFile() BegLine() endif
<CursorDown>  if not Down() BegFile() endif
<Ins>         EndProcess(kADD)
<Alt u>       EndProcess(kUPDATE)
<Alt r>       EndProcess(kRENAME)
<Del>         EndProcess(kDELETE)
<Alt p>       EndProcess(kPalette)
end

string list_footer[] = "{Enter}-Load {Ins}-Add {Alt-U}-Update {Alt-R}-Ren {Del}-Del {Alt-P}-Palette {Esc}-Exit"
//string list_footer[] = "{Enter}-Load {Ins}-Add {Alt-U}-Update {Alt-R}-Ren {Del}-Del {Esc}-Exit"

proc list_startup()
    UnHook(list_startup)
    Enable(list_keys)
    ListFooter(list_footer)
end

integer scheme_buffer = 0

proc Main()
    integer result, cur_id, palette, esc_pressed
    string scheme_file[_MAXPATH_], scheme_name[255]

    cur_id = GetBufferId()
    scheme_file = SearchPath("colors.ini", Query(TSEPath), ".")

    build_scheme_buffer(scheme_buffer, scheme_file)
    palette = FALSE
    esc_pressed = FALSE
    repeat
        if palette
            ExecMacro("setpalette -x")
            palette = FALSE
        endif
        GotoBufferId(scheme_buffer)
        Hook(_LIST_STARTUP_, list_startup)
        Set(Y1, 4)
        BufferVideo()
        result = List("Color Schemes", Max(LongestLineInBuffer(), Length(list_footer) - 12))
        scheme_name = GetText(1, CurrLineLen())
        if  scheme_name == ''
        and result <> kADD
          result = FALSE
        endif
        GotoBufferId(cur_id)
        case result
            // add a new scheme
            when kADD
                if add_scheme(scheme_file)
                    build_scheme_buffer(scheme_buffer, scheme_file)
                endif
            // update an existing scheme using current color settings
            when KUPDATE
                update_scheme(scheme_name, scheme_file)
            // delete a scheme
            when kDELETE
                RemoveProfileSection(scheme_name, scheme_file)
                build_scheme_buffer(scheme_buffer, scheme_file)
            // rename a scheme
            when kRENAME
                if rename_scheme(scheme_name, scheme_file)
                    build_scheme_buffer(scheme_buffer, scheme_file)
                endif
            when kPalette
                palette = TRUE
            // load a scheme
            when TRUE
                load_scheme(scheme_name, scheme_file)
                GotoBufferId(scheme_buffer)
                GotoBufferId(cur_id)
                UpdateDisplay(_ALL_WINDOWS_REFRESH_)
                VirtualToRealScreen()
            // escape pressed
            when FALSE
                esc_pressed = TRUE
        endcase
        UnBufferVideo()
    until esc_pressed
end

