/***********************************************************************
  Calendar    Displays a visual, changeable calendar

  Author:     Richard Hendricks

  Date:       Apr  5, 1993 - original release
              Apr  8, 1993 - 3 months at a time calendar. The Previous and
                             Next Months, always started on the same day
                             of the week as the current month. This has
                             been fixed in this release.
              Aug 30, 1994 - Modified to correct incompatibility with
                             COLORS.  Added a Help Line

              Feb  6, 2002 - Ross Boyd - Added non OEM font compatibility fixes

              Mar 14, 2006 - SEM - Fixed bug in Leap Year calculation - thanks
                             to Paul Bennett for the report.

              Jun 29, 2006 - SEM - Fixed bug regarding conflict of calendar vs.
                             vmatch.  Thanks to Jose Adriano Baltieri for the
                             report.
  Overview:

  Three calendar displays are available -- Medium-Size Single Month,
  Three Months at a Time, and a Small Single Month. (Press <Alt C> to
  toggle between them.)

  A small calendar can be inserted into the current file by pressing
  <Alt I>. Here is a sample:

       Ŀ
       April      5th  1993
       Su Mo Tu We Th Fr Sa
                    1  2  3
        4[ 5] 6  7  8  9 10
       11 12 13 14 15 16 17
       18 19 20 21 22 23 24
       25 26 27 28 29 30   
       

  Keys:
      Next Day                  <CursorUp> or <CursorRight>
      Previous Day              <CursorDown> or <CursorLeft>
      
      Next Month                <+>, <SpaceBar> or <Enter>
      Previous Month            <-> or <BackSpace>
      
      Next Year                 <PgUp>
      Previous Year             <PgDn>
      
      First Day Of Month        <Home>
      Last Day Of Month         <End>
      
      First Day Of Year         <Ctrl Home>
      Last Day Of Year          <Ctrl End>
      
      Today                     <Alt T>
      Change Calendar Type      <Alt C>
      Insert Calendar into Text <Alt I>
      Exit Calendar             <Escape>

 ***********************************************************************/


forward proc PutSingleMonth( integer month,
                             integer day,
                             integer year )

forward proc PutTinySingleMonth( integer month,
                                 integer day,
                                 integer year )

forward proc PutThreeMonths( integer month,
                             integer day,
                             integer year )

forward proc InsertTinySingleMonth( integer month,
                                    integer day,
                                    integer year )

forward proc DisplayCalendar()

Constant SingleMonth = 1
Constant TinySingleMonth = 2
Constant ThreeMonths = 3

integer CurrYear, CurrMonth, CurrDay, CurrDayOfWeek
integer CalType = SingleMonth

integer NormalAttr = Color( Black on White ),
        TodayAttr = Color( Red on White )

string proc DayThString( integer day )
  case day
    when 1, 21, 31
      return( format(day:2)+"st" )
    when 2, 22
      return( format(day:2)+"nd" )
    when 3, 23
      return( format(day:2)+"rd" )
  endcase
  return( format(day:2)+"th" )
end DayThString

string proc MonthNameString( integer month )
  string MonthNames[110] = "January  February March    April    May      June     July     August   SeptemberOctober  November December " // 9 characters each
  return( substr( MonthNames, ((Month-1)*9)+1, 9 ) )
end MonthNameString

integer proc DayOfWeekVal( integer month, integer day, integer year )
  // 1-Sunday, 2-Monday, 3-Tuesday,.. 7-Saturday
  Integer DaysSince

  if(Month < 3)
   year = year - 1
   Month = Month + 12
  endif

  DaysSince = (year / 400 - year / 100 + year / 4 + year + Month * 3)
  if(Month > 4)
      DaysSince = DaysSince - 1
  endif
  if(Month > 6)
      DaysSince = DaysSince - 1
  endif
  if(Month > 9)
      DaysSince = DaysSince - 1
  endif
  if (Month > 11)
      DaysSince = DaysSince - 1
  endif
  DaysSince = DaysSince + Day

  return ( ( DaysSince mod 7 ) + 1 )
end DayOfWeekVal

integer proc isLeapYear(integer year)
    /* from Algorithms for Programmers, Binstock and Rex.
    if year mod 4 <> 0 return (False) endif
    if year < 1582 return (True) endif
    if year mod 100 <> 0 return (True) endif
    if year mod 400 <> 0 return (False) endif
    return (True)
    */
    // from the C FAQ return (year mod 4 == 0 and (year mod 100 <> 0 or year mod 400 == 0)) //c-faq
    // from K&R return ((year mod 4 == 0 and year mod 100 <> 0) or year mod 400 == 0) //k&r
    return (year mod 400 == 0 or (year mod 4 == 0 and year mod 100 <> 0))
end

// old incorrect code
#if 0
integer proc isLeapYear( integer year )
  return( iif( year mod 4 or (not year mod 100 and year mod 400),
                false, true ) )
end isLeapYear
#endif

integer proc DaysInMonthVal( integer month, integer year )
  string MonthDays[24]="312831303130313130313031"

  if month <> 2
    return( val( substr( MonthDays, ((Month-1)*2)+1, 2 ) ) )
  endif
  return( iif( isLeapYear( year ), 29, 28 ) )
end DaysInMonthVal

proc ToggleCalType()
  case CalType
    when SingleMonth
      CalType = TinySingleMonth
    when TinySingleMonth
      CalType = ThreeMonths
    when ThreeMonths
      CalType = SingleMonth
    otherwise
      CalType = SingleMonth
  endcase
  FullWindow()
  UpdateDisplay(_All_Windows_Refresh_)
  DisplayCalendar()
end ToggleCalType

proc DisplayCalendar()
  case CalType
    when SingleMonth
      PutSingleMonth( CurrMonth, CurrDay, CurrYear )
    when TinySingleMonth
      PutTinySingleMonth( CurrMonth, CurrDay, CurrYear )
    when ThreeMonths
      PutThreeMonths( CurrMonth, CurrDay, CurrYear )
    otherwise
      PutSingleMonth( CurrMonth, CurrDay, CurrYear )
  endcase
end DisplayCalendar

proc Today()
  GetDate( CurrMonth, CurrDay, CurrYear, CurrDayOfWeek )
  DisplayCalendar()
end Today

proc NextYear()
  CurrYear = CurrYear + 1
  DisplayCalendar()
end NextYear

proc PrevYear()
  CurrYear = CurrYear - 1
  DisplayCalendar()
end PrevYear

proc NextMonth()
  CurrMonth = CurrMonth + 1
  CurrYear  = iif( CurrMonth > 12, CurrYear+1, CurrYear )
  CurrMonth = iif( CurrMonth > 12, 1, CurrMonth )
  CurrDay = iif( CurrDay > DaysInMonthVal( CurrMonth, CurrYear ),
                           DaysInMonthVal( CurrMonth, CurrYear ), CurrDay )
  DisplayCalendar()
end NextMonth

proc PrevMonth()
  CurrMonth = CurrMonth - 1
  CurrYear  = iif( CurrMonth < 1, CurrYear-1, CurrYear )
  CurrMonth = iif( CurrMonth < 1, 12, CurrMonth )
  CurrDay = iif( CurrDay > DaysInMonthVal( CurrMonth, CurrYear ),
                           DaysInMonthVal( CurrMonth, CurrYear ), CurrDay )
  DisplayCalendar()
end PrevMonth

proc NextDay()
  CurrDay = CurrDay + 1
  if CurrDay > DaysInMonthVal( CurrMonth, CurrYear )
    CurrMonth = CurrMonth + 1
    CurrDay = 1
  endif
  CurrYear  = iif( CurrMonth > 12, CurrYear+1, CurrYear )
  CurrMonth = iif( CurrMonth > 12, 1, CurrMonth )
  DisplayCalendar()
end NextDay

proc PrevDay()
  CurrDay = CurrDay - 1
  if CurrDay < 1
    CurrMonth = CurrMonth - 1
    CurrYear  = iif( CurrMonth < 1, CurrYear-1, CurrYear )
    CurrMonth = iif( CurrMonth < 1, 12, CurrMonth )
    CurrDay = DaysInMonthVal( CurrMonth, CurrYear )
  endif
  DisplayCalendar()
end PrevDay

proc PutSingleMonth( integer month, integer day, integer year )
  integer MonthDayOfWeek, DaysInMonth,
          d, c_day, c_week
  String           // Month        Day         Year
    MonthCal01[40]="͸",  // 36 X 18
    MonthCal02[40]="                                  ",
    MonthCal03[40]=";",
    MonthCal04[40]="Ŀ",
    MonthCal05[40]=" Su  Mo  Tu  We  Th  Fr  Sa ",
    MonthCal06[40]="Ĵ",
    MonthCal07[40]="                            ",
    MonthCal08[40]="Ĵ",
    MonthCal09[40]="                            ",
    MonthCal10[40]="Ĵ",
    MonthCal11[40]="                            ",
    MonthCal12[40]="Ĵ",
    MonthCal13[40]="                            ",
    MonthCal14[40]="Ĵ",
    MonthCal15[40]="                            ",
    MonthCal16[40]="Ĵ",
    MonthCal17[40]="                            ",
    MonthCal18[40]=""

  Window( 23, 3, 23+35, 3+17 )
  VHomeCursor()
  ClrScr()
  Set( Attr, NormalAttr )
  Set( Cursor, Off )
  PutOemStrXY( 1,  1,MonthCal01 )
  PutOemStrXY( 1,  2,MonthCal02 )
  PutOemStrXY( 1,  3,MonthCal03 )
  PutOemStrXY( 1,  4,MonthCal04 )
  PutOemStrXY( 1,  5,MonthCal05 )
  PutOemStrXY( 1,  6,MonthCal06 )
  PutOemStrXY( 1,  7,MonthCal07 )
  PutOemStrXY( 1,  8,MonthCal08 )
  PutOemStrXY( 1,  9,MonthCal09 )
  PutOemStrXY( 1, 10,MonthCal10 )
  PutOemStrXY( 1, 11,MonthCal11 )
  PutOemStrXY( 1, 12,MonthCal12 )
  PutOemStrXY( 1, 13,MonthCal13 )
  PutOemStrXY( 1, 14,MonthCal14 )
  PutOemStrXY( 1, 15,MonthCal15 )
  PutOemStrXY( 1, 16,MonthCal16 )
  PutOemStrXY( 1, 17,MonthCal17 )
  PutOemStrXY( 1, 18,MonthCal18 )
  VGotoXY( 4, 2 )
  Write( MonthNameString( Month ) )
  VGotoXY( 30, 2 )
  Write( Str( Year ) )
  VGotoXY( 17, 2 )
  Write( DayThString( Day ) )  // 1st, 2nd, 3rd, 4th....
  MonthDayOfWeek = DayOfWeekVal( month, 1, year )  // First Day Of Month
  DaysInMonth = DaysInMonthVal( month, year )

  d = 1
  c_day  = MonthDayOfWeek
  c_week = 1
  while d <= DaysInMonth
    VGotoXY( iif( c_day == 1, 3, (c_day*5)-2 ), iif( c_week == 1, 7, (c_week*2)+5 ) )
    Write( Format( d:2 ) )
    if d == Day
      VGotoXY( iif( c_day == 1, 3, (c_day*5)-2 ), iif( c_week == 1, 7, (c_week*2)+5 ) )
      PutAttr( TodayAttr, 2 )
    endif
    c_day = c_day + 1
    if c_day > 7
      c_day = 1
      c_week = c_week + 1
    endif
    d = d + 1
  endwhile
  Set( Cursor, On )
end PutSingleMonth

proc PutTinySingleMonth( integer month, integer day, integer year )
  integer MonthDayOfWeek, DaysInMonth,
          d, c_day, c_week
/*
  123456789  12cc 1234   8 x 20
  Su Mo Tu We Th Fr Sa
                    01
  02 03 04 05 06 07 08
  09 10 11 12 13 14 15
  16 17 18 19 20 21 22
  23 24 25 26 27 28 29
  30 31

*/
  Window( 60, 1, 60+19, 1+7 )
  VHomeCursor()
  ClrScr()
  Set( Attr, NormalAttr )
  Set( Cursor, Off )
  VGotoXY( 1, 1 )
  Write( MonthNameString( Month ) )
  VGotoXY( 17, 1 )
  Write( Str( Year ) )
  VGotoXY( 12, 1 )
  Write( DayThString( Day ) )  // 1st, 2nd, 3rd, 4th....
  VGotoXY( 1, 2 )
  Write( "Su Mo Tu We Th Fr Sa" )
  MonthDayOfWeek = DayOfWeekVal( month, 1, year )  // First Day Of Month
  DaysInMonth = DaysInMonthVal( month, year )

  d = 1
  c_day  = MonthDayOfWeek
  c_week = 1
  while d <= DaysInMonth
    VGotoXY( iif( c_day == 1, 1, (c_day*3)-2 ), c_week+2 )
    Write( Format( d:2 ) )
    if d == Day
      VGotoXY( iif( c_day == 1, 1, (c_day*3)-2 ), c_week+2 )
      PutAttr( TodayAttr, 2 )
    endif
    c_day = c_day + 1
    if c_day > 7
      c_day = 1
      c_week = c_week + 1
    endif
    d = d + 1
  endwhile
  Set( Cursor, On )
end PutTinySingleMonth

proc PutThreeMonths( integer month, integer day, integer year )
  integer MonthDayOfWeek, DaysInMonth,
          d, c_day, c_week,
          w_month, w_year
  String           // Month        Day         Year
    MonthCal01[80]="Ŀ                                  Ŀ",
    MonthCal02[80]="                                                                          ",
    MonthCal03[80]="                                  ",
    MonthCal04[80]="Ŀ                                  Ŀ",
    MonthCal05[80]="SuMoTuWeThFrSa                                  SuMoTuWeThFrSa",
    MonthCal06[80]="͸Ĵ",
    MonthCal07[80]="                                                              ",
    MonthCal08[80]=";Ĵ",
    MonthCal09[80]="              ͻ              ",
    MonthCal10[80]="ĺ Su  Mo  Tu  We  Th  Fr  Sa Ĵ",
    MonthCal11[80]="              Ķ              ",
    MonthCal12[80]="ĺ                            Ĵ",
    MonthCal13[80]="              Ķ              ",
    MonthCal14[80]="ĺ                            Ĵ",
    MonthCal15[80]="              Ķ              ",
    MonthCal16[80]="ĺ                            Ĵ",
    MonthCal17[80]="              Ķ              ",
    MonthCal18[80]="ĺ                            ",
    MonthCal19[80]="                     Ķ",
    MonthCal20[80]="                                                 ",
    MonthCal21[80]="                     Ķ",
    MonthCal22[80]="                                                 ",
    MonthCal23[80]="                     ͼ"

  Window( 2, 1, 79, 23 )
  VHomeCursor()
  ClrScr()
  Set( Cursor, Off )
  Set( Attr, NormalAttr )
  PutOemStrXY( 1,  1, MonthCal01 )
  PutOemStrXY( 1,  2, MonthCal02 )
  PutOemStrXY( 1,  3, MonthCal03 )
  PutOemStrXY( 1,  4, MonthCal04 )
  PutOemStrXY( 1,  5, MonthCal05 )
  PutOemStrXY( 1,  6, MonthCal06 )
  PutOemStrXY( 1,  7, MonthCal07 )
  PutOemStrXY( 1,  8, MonthCal08 )
  PutOemStrXY( 1,  9, MonthCal09 )
  PutOemStrXY( 1, 10, MonthCal10 )
  PutOemStrXY( 1, 11, MonthCal11 )
  PutOemStrXY( 1, 12, MonthCal12 )
  PutOemStrXY( 1, 13, MonthCal13 )
  PutOemStrXY( 1, 14, MonthCal14 )
  PutOemStrXY( 1, 15, MonthCal15 )
  PutOemStrXY( 1, 16, MonthCal16 )
  PutOemStrXY( 1, 17, MonthCal17 )
  PutOemStrXY( 1, 18, MonthCal18 )
  PutOemStrXY( 1, 19, MonthCal19 )
  PutOemStrXY( 1, 20, MonthCal20 )
  PutOemStrXY( 1, 21, MonthCal21 )
  PutOemStrXY( 1, 22, MonthCal22 )
  PutOemStrXY( 1, 23, MonthCal23 )

// Display Current Month

  VGotoXY( 25, 7 )
  Write( MonthNameString( Month ) )
  VGotoXY( 51, 7 )
  Write( Str( Year ) )
  VGotoXY( 38, 7 )
  Write( DayThString( Day ) )  // 1st, 2nd, 3rd, 4th....
  MonthDayOfWeek = DayOfWeekVal( month, 1, year )  // First Day Of Month
  DaysInMonth = DaysInMonthVal( month, year )

  d = 1
  c_day  = MonthDayOfWeek
  c_week = 1
  while d <= DaysInMonth
    VGotoXY( iif( c_day == 1, 24, 21+(c_day*5)-2 ), iif( c_week == 1, 12, (c_week*2)+10 ) )
    Write( Format( d:2 ) )
    if d == Day
      VGotoXY( iif( c_day == 1, 24, 21+(c_day*5)-2 ), iif( c_week == 1, 12, (c_week*2)+10 ) )
      PutAttr( TodayAttr, 2 )
    endif
    c_day = c_day + 1
    if c_day > 7
      c_day = 1
      c_week = c_week + 1
    endif
    d = d + 1
  endwhile

// Display Previous Month

  w_month = month - 1
  w_year  = year
  w_year  = iif( w_month < 1, w_year - 1, w_year )
  w_month = iif( w_month < 1, 12, w_month )
  VGotoXY( 3, 2 )
  Write( MonthNameString( w_month ) )
  VGotoXY( 17, 2 )
  Write( Str( w_year ) )
  MonthDayOfWeek = DayOfWeekVal( w_month, 1, w_year )  // First Day Of Month
  DaysInMonth = DaysInMonthVal( w_month, w_year )

  d = 1
  c_day  = MonthDayOfWeek
  c_week = 1
  while d <= DaysInMonth
    VGotoXY( iif( c_day == 1, 2, (c_day*3)-1 ), iif( c_week == 1, 7, (c_week*2)+5 ) )
    Write( Format( d:2 ) )
    c_day = c_day + 1
    if c_day > 7
      c_day = 1
      c_week = c_week + 1
    endif
    d = d + 1
  endwhile

// Display Next Month

  w_month = month + 1
  w_year  = year
  w_year  = iif( w_month > 12, w_year + 1, w_year )
  w_month = iif( w_month > 12, 1, w_month )
  VGotoXY( 59, 2 )
  Write( MonthNameString( w_month ) )
  VGotoXY( 73, 2 )
  Write( Str( w_year ) )
  MonthDayOfWeek = DayOfWeekVal( w_month, 1, w_year )  // First Day Of Month
  DaysInMonth = DaysInMonthVal( w_month, w_year )

  d = 1
  c_day  = MonthDayOfWeek
  c_week = 1
  while d <= DaysInMonth
    VGotoXY( iif( c_day == 1, 58, 56+(c_day*3)-1 ), iif( c_week == 1, 7, (c_week*2)+5 ) )
    Write( Format( d:2 ) )
    c_day = c_day + 1
    if c_day > 7
      c_day = 1
      c_week = c_week + 1
    endif
    d = d + 1
  endwhile
  Set( Cursor, On )
end PutThreeMonths

proc InsertTinySingleMonth( integer month, integer day, integer year )
  integer MonthDayOfWeek, DaysInMonth,
          d, c_day, c_week,
          scol
/*
 +--------------------+
 |123456789  12cc 1234|
 |Su Mo Tu We Th Fr Sa|
 |                  01|
 |02 03 04 05 06 07 08|
 |09 10 11 12 13 14 15|
 |16 17 18 19 20 21 22|
 |23 24 25 26 27 28 29|
 |30 31               |
 +--------------------+
*/
  scol = CurrCol()
  PushPosition()
  GotoColumn( scol )
  InsertText( "+--------------------+ ", _INSERT_ )
  if not Down()
    AddLine()
  endif
  GotoColumn( scol )
  InsertText( "|                    | ", _INSERT_ )
  GotoColumn( scol+1 )
  InsertText( MonthNameString( Month ), _OVERWRITE_ )
  GotoColumn( scol+17 )
  InsertText( Str( Year ), _OVERWRITE_ )
  GotoColumn( scol+11 )
  InsertText( DayThString( Day ), _OVERWRITE_ )  // 1st, 2nd, 3rd, 4th....
  if not Down()
    AddLine()
  endif
  GotoColumn( scol )
  InsertText( "|Su Mo Tu We Th Fr Sa| ", _INSERT_ )
  MonthDayOfWeek = DayOfWeekVal( month, 1, year )  // First Day Of Month
  DaysInMonth = DaysInMonthVal( month, year )

  d = 1
  c_day  = MonthDayOfWeek
  c_week = 1
  if not Down()
    AddLine()
  endif
  GotoColumn( scol )
  InsertText( "|                    | ", _INSERT_ )
  while d <= DaysInMonth
    GotoColumn( iif(c_day == 1, 2, (c_day*3)-1 ) + scol - 1 )
    InsertText( Format( d:2 ), _OVERWRITE_ )
    if d == Day
      GotoColumn( iif(c_day == 1, 1, (c_day*3)-2 ) + scol - 1 )
      InsertText( "["+Format( d:2 )+"]", _OVERWRITE_ )
    endif
    c_day = c_day + 1
    if c_day > 7
      c_day = 1
      c_week = c_week + 1
      if not Down()
        AddLine()
      endif
      GotoColumn( scol )
      InsertText( "|                    | ", _INSERT_ )
    endif
    d = d + 1
  endwhile
  if not Down()
    AddLine()
  endif
  GotoColumn( scol )
  InsertText( "+--------------------+ ", _INSERT_ )
  PopPosition()
end InsertTinySingleMonth

proc mBreakHookChain()
  BreakHookChain()
end mBreakHookChain

keydef CalKeys
  <Alt T>             Today()
  <Alt C>             ToggleCalType()
  <Alt I>             InsertTinySingleMonth( CurrMonth, CurrDay, CurrYear )
                      FullWindow()
                      UpdateDisplay(_All_Windows_Refresh_)
                      DisplayCalendar()

  <PgDn>              PrevYear()
  <PgUp>              NextYear()

  <Grey+>             NextMonth()
  <Shift =>           NextMonth()
  <SpaceBar>          NextMonth()
  <Enter>             NextMonth()

  <Grey->             PrevMonth()
  <->                 PrevMonth()
  <BackSpace>         PrevMonth()

  <CursorRight>       NextDay()
  <CursorUp>          NextDay()
  <CursorLeft>        PrevDay()
  <CursorDown>        PrevDay()

  <Home>              CurrDay = 1  // First Day Of Month
                      DisplayCalendar()

  <End>               CurrDay = DaysInMonthVal( CurrMonth, CurrYear )  // Last Day Of Month
                      DisplayCalendar()

  <Ctrl Home>         CurrMonth = 1 // First Day Of January
                      CurrDay = 1
                      DisplayCalendar()

  <Ctrl End>          CurrMonth = 12 // Last Day Of December
                      CurrDay = 31
                      DisplayCalendar()

  <Escape>            FullWindow()
                      UnHook( mBreakHookChain )
                      UpdateDisplay(_All_Windows_Refresh_)
                      DisAble( CalKeys )

  <HelpLine> "{Dy} <--> {Mo} +/- {Yr} Page {Mo B/E} Home/End {Yr B/E} ^Hom/^En {Today} @T {Chng} @C {Insrt} @I"
end CalKeys

proc MAIN()
  GetDate( CurrMonth, CurrDay, CurrYear, CurrDayOfWeek )
  Hook( _AFTER_UPDATE_DISPLAY_, mBreakHookChain )
  Hook( _POST_UPDATE_ALL_WINDOWS_, mBreakHookChain )
  UpdateDisplay()
  Enable( CalKeys, _EXCLUSIVE_ )
  DisplayCalendar()
end MAIN
// end-of-file CALENDAR.S
