/*************************************************************************
  Book        Bookmark manager - allows visual display of active bookmarks

  Author:     SemWare
              (Based on macros by Jim Susoy and Howard Kapustein)

  06 Mar 2024  - Eckhard Hillmann
      If you use the *book.s* macro to manage your bookmarks, you
      should add two lines in there.
      Both fix old problems that occur when *book* is used in
      combination with *whtspc*.

      If you don't add them this most likely will happen:

      1 - When you delete a bookmark which is currently visible
          on the screen the line marking will only be removed
          from the screen when the screen is moved by for example
          scroll-up/-down, roll-up/-down, page-up/-down...

      2 - When you have a loaded file split in two or more
          windows which each show nearly the same area of text on
          the screen and then set a bookmark on a line which is
          visible in every window, only the active window shows
          the newly set mark. The other windows only shows it
          when you switch to them.

      At the end of ViewBookMarks():
        Add an UpdateDisplay
      Just before the Message in SetBookMark():
        Add an UpdateDisplay

  27 Jan 2013  - SEM - bug fix contributed by Knud
      Van Eeden, routine AddInfoToList() - the Format() was such
      that line numbers > 1 million were truncated. Fixed, by
      changing from Format(...6) to Format(...10)

  Jun 18, 1999 - SEM - bug fix contributed by Christian Pradel,
      routine ViewBookMarks()

  May 29, 1997 - (Kyle Watkins) Various changes and enhancements:

     The 'last_book_set' and 'last_book_visited' values are set
     to the highest existing bookmark letter (or highest letter
     if no bookmarks exist) when loading the macro (in case
     bookmarks were set prior to loading macro).

     The cursor positions in the files containing bookmarks are
     now preserved while the bookmark view is generated.

     Setting new bookmarks only overwrites an existing bookmark
     if no "unused" bookmark positions exist.

     Added:

     Within the ViewBookMarks() List:

     - For bookmarks that are contained in the current file, a
       character (denoted by 'char_curr_file') is displayed in
       column 1.

     - The default cursor line on start-up of the ViewBookMarks()
       List is the bookmark specified by 'last_book_visited', or
       the previous mark to this if 'last_book_visited' has been
       deleted.

     - The last_book_set character is uppercased in the list

     - The bookmark letter of the default start-up cursor line
       (as explained above) and the current line number of the
       current file are displayed in the list header.

     - If a bookmark is selected to go to from the list,
       'last_book_visited' is set to that bookmark

     - The bookmark data text can be scrolled. See the keydef
       ListKeys for key assignments.

  Jan 31, 1997 - Documentation update
  Aug 29, 1995 - Use alpha's on display rather than numbers (GDB)
  Jul 19, 1995 - Rewritten for TSE 2.5 (Sammy Mitchell)
  Jun 22, 1994 - More enhancements
  Apr 23, 1993 - Enhancements from Ray Asbury
  Oct 10, 1992 - Initial version (Sammy Mitchell)

  Overview:

  These macros will give a visual display of all active bookmarks.
  This comes in handy when you cannot remember which bookmark takes
  you to the desired location.  When you pop up the PickList, a
  portion of the line where bookmarks are located will be displayed so
  you can choose based on the content of the line.

  Additionally, bookmarks can be thought of to reside in a ring, and
  you can easily visit each bookmark by using the GotoNextBookMark and
  GotoPreviousBookMark commands.

  Keys:
            <Ctrl />        PlaceBookMark
            <Ctrl .>        GotoNextBookMark
            <Ctrl ,>        GotoPreviousBookMark
            <CtrlShift />   ViewBookMarkList

  Usage notes:

  Using the <Ctrl /> key, you can place bookmarks at any place in any
  file.  The bookmarks are lettered in alphabetical order beginning
  with a.  Pressing the <Ctrl .> and <Ctrl ,> keys will go to the next
  and previous bookmarks respectively.  Pressing the <CtrlShift /> key
  will invoke the PickList which shows the bookmark number, file, line
  number, and part of the text on the line where each bookmark is
  located.

  Copyright 1992-1997 SemWare Corporation.  All Rights Reserved Worldwide.

  Use, modification, and distribution of this SAL macro is encouraged
  by SemWare provided that this statement, including the above
  copyright notice, is not removed; and provided that no fee or other
  remuneration is received for distribution.  You may add your own
  copyright notice to cover new matter you add to the macro, but
  SemWare Corporation will neither support nor assume legal
  responsibility for any material added or any changes made to the
  macro.

*************************************************************************/

constant
    NUM_BOOKMARKS = 26,
    BOOKCOL       = 3,    //list column which contains bookmark letter
    INFOCOL       = 26    //list column where the bookmark data starts

integer bookmark_buffer,
        scroll_count,     //current number of characters scrolled right
        max_scroll_count, //maximum number of characters to scroll right
        max_list_line     //length of longest line in ViewBookMarks() list

string
    last_book_set[] = 'z',
    last_book_visited[] = 'z'

string
    char_curr_file[1] = "~"     // character used for bookmark of currfile

/**************************************************************************
  The following procedures deal with a circular set of bookmarks covering
  the range of Asc('a') through Asc('z').
 **************************************************************************/

/**************************************************************************
  IncBookMark() returns the next valid bookmark, in sequence, given the
  current bookmark.

  NextBookMark() returns the next 'set' bookmark in sequence, given the
  current bookmark.

  PrevBookMark() returns the previous 'set' bookmark in sequence, given the
  current bookmark.

 **************************************************************************/

string proc IncBookMark(string bookmark)
    return (iif(bookmark >= 'z', 'a', Chr(Asc(bookmark) + 1)))
end

string proc NextBookMark(string bkmrk)
    string bookmark[1] = bkmrk

    do NUM_BOOKMARKS times
        bookmark = IncBookMark(bookmark)
        if isBookMarkSet(bookmark)
            break
        endif
    enddo
    return (bookmark)
end

string proc PrevBookMark(string bkmrk)
    string bookmark[1] = bkmrk
    do NUM_BOOKMARKS times
        bookmark = Chr(Asc(bookmark) - 1)
        if bookmark < 'a'
            bookmark = 'z'
        endif
        if isBookMarkSet(bookmark)
            break
        endif
    enddo
    return (bookmark)
end

/**************************************************************************
  GetBookMarkFromLine() returns the bookmark string, gathered from the
  current line of the list that is generated by ViewBookMarks().

  RemoveBookmark() removes the selected bookmark.

  SetLastVisit() sets the 'last_book_visited' global variable and returns
  the bookmark character that was used to set 'last_book_visited'
 **************************************************************************/

string proc GetBookMarkFromLine()
    return (iif(CurrLineLen(), Lower(GetText(BOOKCOL, 1)), ""))
end

proc RemoveBookmark()
    string bookmark[1] = GetBookMarkFromLine()

    if Length(bookmark)
        DelBookmark(bookmark)
        KillLine()
    endif
end

string proc SetLastVisit()
    if CurrLineLen()
        last_book_visited = GetBookMarkFromLine()
        return (last_book_visited)
    endif
    return("")
end

/**************************************************************************

  The following procedures are used to scroll the bookmark data within the
  ViewBookMarks() list

  scroll_count keeps track of the number of columns that have been scrolled
  right

  max_scroll_count is the limit scroll_count can reach and is based on the
  longest line in the ViewBookMarks() list and the width of the
  ViewBookMarks() list window

  GetLongLine() returns the longest line in the list generated by
  ViewBookMarks()

  ScrollDataRight() scrolls bookmark data one position right if scroll_count
  is less than max_scroll_count

  ScrollDataLeft() scrolls bookmark data one position left if scroll_count
  is greater than zero

  ScrollDataHome() scrolls bookmark data to the left until scroll_count = 0

  ScrollDataEnd() scrolls bookmark data to the right until the end of the
  current bookmark data line can be displayed in the list

  ScrollAllDataEnd() scrolls bookmark data to the right until the end of the
  longest bookmark data line can be displayed in the list

 **************************************************************************/

integer proc GetLongLine()
    integer longline = 0
    PushPosition()
    BegFile()
    repeat
        longline = Max(longline, CurrLineLen())
    until not Down()
    PopPosition()
    return(longline)
end

proc ScrollDataRight()
    integer saveline
        if (scroll_count < max_scroll_count)
            saveline = CurrLine()
            PushBlock()
            UnMarkBlock()
            MarkColumn(1,INFOCOL,NumLines(),INFOCOL)
            GotoLine(1)
            GotoColumn(max_list_line + 1)
            MoveBlock()
            UnMarkBlock()
            PopBlock()
            scroll_count = scroll_count + 1
            GotoLine(saveline)
        endif
end

proc ScrollDataLeft()
    integer saveline
        if scroll_count > 0
            saveline = CurrLine()
            PushBlock()
            UnMarkBlock()
            MarkColumn(1,max_list_line,NumLines(),max_list_line)
            GotoLine(1)
            GotoColumn(INFOCOL)
            MoveBlock()
            UnMarkBlock()
            PopBlock()
            scroll_count = scroll_count - 1
            GotoLine(saveline)
        endif

end

proc ScrollDataHome()
    while scroll_count
        ScrollDataLeft()
    endwhile
end

proc ScrollDataEnd()
    integer amount
    ScrollDataHome()
    amount = PosLastNonWhite() - Query(windowcols)
        while amount > 0
            ScrollDataRight()
            amount = amount - 1
        endwhile
end

proc ScrollAllDataEnd()
    integer amount
    amount = max_list_line - Query(windowcols)
    if amount > 0
        ScrollDataHome()
        while amount > 0
            ScrollDataRight()
            amount = amount - 1
        endwhile
    endif
end

keydef ListKeys
<Del>           RemoveBookmark()
<CursorLeft>    ScrollDataLeft()
<CursorRight>   ScrollDataRight()
<Home>          ScrollDataHome()
<Ctrl Home>     ScrollDataHome()
<End>           ScrollDataEnd()
<Ctrl End>      ScrollAllDataEnd()
end

proc ListStartup()
    Unhook(ListStartup)
    if Enable(ListKeys)
        ListFooter("{Enter}-Goto {Del}-Remove {Escape}-Cancel")
    endif
end

/**************************************************************************
  AddInfoToList() adds information about a particular bookmark to the
  'list' buffer that is created by ViewBookMarks().  The format of the
  'information line' is of the form:

a[X] FILENAME      LLLLL# 'INFORMATION ON CURRENT BOOKMARKED LINE'


    "a" is either a space character or the character specified by
        'char_curr_file'.  If the particular bookmark entry is in the
        current file, the 'char_curr_file' character is used, otherwise a
        space is used.

    "[X]" represents the bookmark (X is replaced by a-z). If the
          bookmark is the last_book_set, it is uppercased.

    "FILENAME" is the name of the file containing the bookmark (Only the
               first 12 characters of filename are used).

    "LLLLL#" represents the line number that is bookmarked.

    The remainder of the line is data taken from the bookmarked line.

  ViewBookMarks() generates a 'list' of currently 'set' bookmarks using
  the above format.  Additionally:

    - The default cursor line on start-up of the ViewBookMarks() List is
      the bookmark specified by 'last_book_visited', or the line
      containing the previous mark if 'last_book_visited' has been
      deleted.

    - The bookmark letter of the default start-up cursor line (as
      explained above) and the current line number of the current file are
      displayed in the list header.

 **************************************************************************/

proc AddInfoToList(string bookmark, integer fileid)
        AddLine(Format(iif(fileid == GetBufferId(),char_curr_file," ")+
                "["+ (iif(bookmark == last_book_set,Upper(bookmark),
                bookmark))+ "]"; SplitPath(CurrFilename(),_NAME_|_EXT_)
                :-12;CurrLine():10; GetText(Max(1,PosFirstNonWhite()),
                CurrLineLen())), bookmark_buffer)
end

proc ViewBookMarks()
    integer bookmark, bookid, temp, fileid, fline

    PushPosition()
    fileid = GetBufferId()
    fline  = CurrLine()
    if not GotoBufferId(bookmark_buffer)
        bookmark_buffer = CreateTempBuffer()
    else
        EmptyBuffer()
    endif
    for bookmark = Asc('a') to Asc('z')
        if isBookMarkSet(Chr(bookmark))
            GetBookMarkInfo(Chr(bookmark),bookid,temp,temp,temp,temp)
            if bookid and bookmark_buffer
                GotoBufferId(bookid)
                PushPosition()
                GotoMark(Chr(bookmark))
                AddInfoToList(Chr(bookmark),fileid)
                PopPosition()
            endif
        endif
    endfor
    GotoBufferId(bookmark_buffer)
    BegFile()
    GotoColumn(BOOKCOL)
    // Bookmark might be upper, force it to lower
    if Lower(Chr(CurrChar())) > last_book_visited
        GotoLine(NumLines())
    else
        repeat
            if Lower(Chr(CurrChar())) > last_book_visited
                Up()
                break
            endif
        until not Down()
    endif
    Hook(_LIST_STARTUP_, ListStartup)
    scroll_count = 0
    max_list_line = GetLongLine()
    max_scroll_count = max_list_line - Query(windowcols)
    if List("["+iif(CurrChar() < 0,"",Chr(CurrChar()))+"]"+" Bookmarks - "+
            str(fline), Query(ScreenCols)) and GotoMark(SetLastVisit())
        KillPosition()
    else
        PopPosition()
    endif
    EmptyBuffer(bookmark_buffer)
    UpdateDisplay(_ALL_WINDOWS_REFRESH_)
end

/**************************************************************************
  ResetMarkPos() returns the next available 'unused' bookmark, or the
  passed bookmark if an 'unused' one is not available.

  GotoNextBookMark() takes you to the next bookmark (based on the global
  variable 'last_book_visited').

  GotoPrevBookMark() takes you to the previous bookmark (based on the global
  variable 'last_book_visited').

  SetBookMark() 'sets' a bookmark using the global 'last_book_set' variable.
  SetBookMark() overwrites an existing bookmark ONLY if an unused bookmark
  position is not found.
 **************************************************************************/

string proc ResetMarkPos(string currmarkpos)   // procedure for finding next
    string i[1] = currmarkpos                  // 'unused' bookmark position,
                                               // if it exists
    repeat
        if isBookMarkSet(i)
            i = IncBookMark(i)
        else
            return (i)
        endif
    until i == currmarkpos
    return (currmarkpos)
end

proc GotoNextBookMark()
    last_book_visited = NextBookMark(last_book_visited)
    GotoMark(last_book_visited)
end

proc GotoPrevBookMark()
    last_book_visited = PrevBookMark(last_book_visited)
    GotoMark(last_book_visited)
end

proc SetBookMark()
    last_book_set = IncBookMark(last_book_set)
    if isBookMarkSet(last_book_set)                  // If mark is 'in-use'
        last_book_set = ResetMarkPos(last_book_set)  // find 'unused' mark,
    endif                                            // if it exists
    if PlaceMark(last_book_set)
        UpdateDisplay(_ALL_WINDOWS_REFRESH_)
        Message("Bookmark [", last_book_set, "] Set")
    endif
end

/**************************************************************************

Bookmarks may already be set before loading this macro, so on the initial
'load', we scan the existing bookmarks, starting at the highest possible
bookmark position ("z" down through "a"), and set the global 'last_book_set'
to highest 'in-use' mark found.

 **************************************************************************/

proc whenloaded()
    integer i

    last_book_set       = 'z'
    last_book_visited   = 'z'
    for i = Asc('z') downto Asc('a')
        if isBookMarkSet(Chr(i))
            last_book_set = Chr(i)         // set global 'last' bookmark position
            last_book_visited = Chr(i)
            return()
        endif
    endfor
end


<ctrl />        SetBookMark()
<ctrl ,>        GotoPrevBookMark()
<ctrl .>        GotoNextBookMark()
<ctrlshift />   ViewBookMarks()

