import requests
import sys
from datetime import datetime

# Change this to the directory and file name you want your file output to
OUTPUT_FILE = r"/mystic/themes/default/text/dp_topdoors.ans"

##################################################################
# DO NOT EDIT BELOW THIS LINE!
##################################################################

# API endpoint
API_URL = "http://bbs.throwbackbbs.com:5999/query"

def fetch_top_played(limit=10):
    """Fetch top played doors from the API."""
    params = {'cmd': 'top_played', 'limit': limit}
    try:
        response = requests.get(API_URL, params=params)
        if response.status_code == 200:
            return response.json()
        else:
            print("Error fetching data:", response.status_code)
            return {}
    except requests.exceptions.RequestException as e:
        print("Error connecting to API:", e)
        return {}

def format_date(date_str):
    """Convert ISO date to MM/DD/YYYY with proper colored formatting."""
    try:
        dt = datetime.strptime(date_str, "%Y-%m-%d %H:%M:%S")
        # Format the date: Numbers in light green, slashes in dark gray
        formatted_date = "{0}\033[1;30m/\033[1;32m{1}\033[1;30m/\033[1;32m{2}".format(
            dt.strftime("\033[1;32m%m"),
            dt.strftime("%d"),
            dt.strftime("%Y")
        )
        return formatted_date
    except ValueError:
        return "\033[1;30mUnknown\033[0m"

def write_to_ans(data):
    """Write top played stats to an ANSI file."""
    # Use the full path defined earlier
    file_path = OUTPUT_FILE

    # Sort the data by total plays
    sorted_data = sorted(
        data.iteritems(),  # For Python 2.7
        key=lambda x: x[1].get('total_plays', 0),
        reverse=True
    )

    # Determine padding for alignment based on the number of entries
    max_index_len = len(str(len(sorted_data)))

    # Find the max widths for Plays fields
    max_plays_len = max(len("Plays ({0})".format(stats.get('total_plays', 0))) for _, stats in sorted_data)

    # Header
    header_name = "TOP PLAYED FOR {0}".format(datetime.now().strftime("%B %Y")).upper()

    with open(file_path, 'w') as f:
        f.write("\033[1;32m")  # Bright green for header
        f.write("\n " + "{0} DOORPARTY DOORS".format(header_name).center(78) + "\n")
        f.write("\033[0m")  # Reset color

        for index, (door, stats) in enumerate(sorted_data, start=1):
            total_plays = stats.get('total_plays', 0)
            last_played = format_date(stats.get('last_accessed', "Unknown"))

            # Generate fields
            plays_field = "\033[0;32mPlays \033[1;30m(\033[1;32m{0}\033[1;30m)".format(total_plays)
            date_field = "\033[0;32mLast Played \033[1;30m(\033[1;32m{0}\033[1;30m)".format(last_played)

            # Calculate dynamic dashes for Plays
            plays_spacing = max_plays_len - len("Plays ({0})".format(total_plays))
            plays_separator = " \033[1;30m----" + ("-" * plays_spacing) + "\033[0m "

            # Adjust spacing for index alignment
            index_padding = " " * (max_index_len - len(str(index)))

            # Write formatted lines
            f.write(" \033[0;35m{0}{1}{2}\033[1;30m - \033[0m\033[1;35m{3}\033[0m\033[1;30m:\033[0m\n".format(
                index_padding,
                index,
                "",
                door
            ))
            f.write("      {0}{1}{2}\n".format(
                plays_field,
                plays_separator,
                date_field
            ))
        print("Saved to {0}".format(file_path))

def main():
    """Main client script."""
    # Default limit
    limit = 10

    # Check if a limit was provided
    if len(sys.argv) > 1:
        try:
            limit = int(sys.argv[1])
        except ValueError:
            print("Invalid number provided. Using default limit of 10.")

    print("Fetching top {0} most played doors...".format(limit))
    data = fetch_top_played(limit)

    # Call write_to_ans with only the data argument
    write_to_ans(data)

if __name__ == "__main__":
    main()
