REM Program: Hex Editor v8.6a, Module 1 of 6, PD 2024.
REM Author: Erik Jon Oredson
REM Release: 01/20/2024.
REM Status: Public Domain.
REM Email: eoredson@gmail.com
REM Urls: www.filegate.net

' get include file.
REM $INCLUDE: 'hexedit.inc'

' increase stack for recursion.
STACK 4096

' declare error trap.
ON ERROR GOTO Error.Routine

' undocumented:
'   exits program if redirected input detected.
DO
   X$ = INKEY$
   IF X$ = Nul THEN
      EXIT DO
   END IF
LOOP

' increase PSP file table.
Call Increase.Files

' initialize variables.
CALL InitVars

' initialize server name.
CALL StoreServerName

' read config file.
CALL ReadConfigFile(Filename)

' default windows detected.
Windows.Detected = False

' check windows override.
IF Load.Windows THEN
   ' removed 02/20/2024
   'Windows.Detected = True
END IF

' check DOS override.
IF Load.DOS THEN
   Windows.Detected = False
END IF

' get current drive/directory.
CALL GetDrive(Drive.Number, Directory$)

' dimension multiple file arrays.
REDIM MultiFilename1(1 TO 5) AS STRING
REDIM MultiFilenames(1 TO 15) AS STRING

' init undo/marker file.
CALL OpenUndoFile(VarQ)

' check bad filename.
IF VarQ THEN
   GOTO Error.Exit
END IF

' init paste file.
CALL InitializePaste(VarQ)

' check bad filename.
IF VarQ THEN
   GOTO Error.Exit
END IF

' dimension multiple file array structure.
REDIM File(1 TO 15) AS FileType

' dimension multiple file name array.
REDIM TempFiles(1 TO 15) AS STRING

' dimension menu area arrays.
REDIM Area1(24, 44) AS INTEGER
REDIM Area2(24, 44) AS INTEGER

' reset menu filename flag.
IsMenuFile = False

' check startup config edit file.
IF Filename <> Nul THEN
   CommandLine$ = Filename
   GOTO StartFile
END IF

' check command line.
SELECT CASE COMMAND$
CASE "/?", "-?"
   GOTO BootUsage
END SELECT

' read lowercase command line.
CALL ReadCommandLine(Var$)

' check command line filename.
CommandLine$ = Var$
IF CommandLine$ <> Nul THEN
   GOTO StartFile
END IF

' filename entry starts here.
BeginFile:

' start mouse.
CALL SMouse

' read information from file menu box.
CALL Menu(NetPath2$, Drive$, Directory$, FileSpec1$, FileSpec2$)

' store current menu box netpath.
NetPath$ = NetPath2$

' reset menu filename flag.
IsMenuFile = True

' close menu directory/filename sort files.
CLOSE #2, #3

' check filename.
IF FileSpec2$ = Nul THEN
   ' verify a file open already.
   IF NumberFiles >= 1 THEN
      ASCIIZ = File(CurrentFile).ASCIIZ
      GOTO TopProgram
   END IF

   ' exit program.
   GOTO BootUsage
END IF

' construct filename from menu box.
IF NetPath$ = Nul THEN
   IF Directory$ = Nul THEN
      Filename = Drive$ + ":\" + FileSpec2$
   ELSE
      Filename = Drive$ + ":\" + Directory$ + "\" + FileSpec2$
   END IF
ELSE
   IF Directory$ = Nul THEN
      Filename = NetPath$ + "\" + FileSpec2$
   ELSE
      Filename = NetPath$ + "\" + Directory$ + "\" + FileSpec2$
   END IF
END IF

' store filename from menu box.
CommandLine$ = Filename

' filename entry loop starts here.
StartFile:

' restart mouse.
CALL SMouse

' start filename entry loop.
DO
   ' check filenames
   IF CommandLine$ = Nul THEN
      EXIT DO
   END IF

   ' get filename from command line.
   CommandLine$ = RTRIM$(CommandLine$)
   CommandLine$ = LTRIM$(CommandLine$)

   ' check menu filename returned.
   IF IsMenuFile THEN
      Filename = CommandLine$
      CommandLine$ = Nul
   END IF

   ' parse command line
   IF IsMenuFile = False THEN
      IF LEFT$(CommandLine$, 1) = Quote THEN
         Var = INSTR(2, CommandLine$, Quote)
         IF Var = False THEN
            GOTO BeginFile
         END IF
         Filename = LEFT$(CommandLine$, Var - 1)
         Filename = MID$(Filename, 2)
         IF Filename = Nul THEN
            GOTO BeginFile
         END IF
         CommandLine$ = MID$(CommandLine$, Var + 1)
      ELSE
         Var = INSTR(CommandLine$, " ")
         IF Var THEN
            Filename = LEFT$(CommandLine$, Var - 1)
            CommandLine$ = MID$(CommandLine$, Var + 1)
         ELSE
            Filename = CommandLine$
            CommandLine$ = Nul
         END IF
      END IF
   END IF

   ' trim filename.
   Filename = RTRIM$(Filename)
   Filename = LTRIM$(Filename)

   ' check long filename in quotes.
   IF LEFT$(Filename, 1) = Quote THEN
      IF RIGHT$(Filename, 1) = Quote THEN
         Filename = MID$(Filename, 2)
         Filename = LEFT$(Filename, LEN(Filename) - 1)
      END IF
   END IF

   ' check max files.
   IF NumberFiles = MaxFiles THEN
      EXIT DO
   END IF

   ' check filenames.
   Var3$ = Filename
   DO
      ' check multiple filenames.
      Var3 = INSTR(Var3$, CHR$(13))
      IF Var3 THEN
         Var$ = LEFT$(Var3$, Var3 - 1)
         Var3$ = MID$(Var3$, Var3 + 1)
      ELSE
         IF LEN(Var3$) THEN
            Var$ = Var3$
            Var3$ = Nul
         ELSE
            EXIT DO
         END IF
      END IF

      ' check filelist specifier.
      IF LEFT$(Var$, 1) = "@" THEN
         CLOSE #1

         ' check filelist filename.
         Var4$ = UCASE$(MID$(Var$, 2))
         IF Windows.Detected THEN
            X$ = DIRz$(Var4$)
         ELSE
            X$ = DIR$(Var4$)
         END IF
         IF Windows.Detected THEN
            ' close long filename search
            InregsX.AX = &H71A1
            InregsX.BX = Wfile.Handle
            CALL InterruptX(&H21, InregsX, OutregsX)
         END IF
         IF X$ <> Nul THEN

            ' ambiguate filename.
            ASCIIZ3 = ASCIIZ
            ASCIIZ = Var4$ + CHR$(0)
            CALL GetShortFilename(Short.Filename$)
            ASCIIZ = ASCIIZ3

            ' store ambiguated filename.
            Var4$ = Short.Filename$

            ' open filelist filename.
            IF LEN(Var4$) THEN
               ErrorTrap = False
               OPEN Var4$ FOR INPUT AS #1
               IF ErrorTrap = False THEN

                  ' read filelist files.
                  DO UNTIL EOF(1)
                     LINE INPUT #1, Var$
                     Var$ = RTRIM$(Var$)
                     Var$ = LTRIM$(Var$)
                     IF LEN(Var$) THEN

                        ' check max files.
                        IF NumberFiles = MaxFiles THEN
                           EXIT DO
                        END IF

                        ' load file.
                        GOSUB LoadMultipleFile
                     END IF
                  LOOP
               END IF
            END IF
         END IF
      ELSE
         ' check max files.
         IF NumberFiles = MaxFiles THEN
            RETURN
         END IF

         ' load file.
         GOSUB LoadMultipleFile
      END IF
   LOOP
LOOP

' check files loaded.
IF NumberFiles = False THEN
   GOTO BeginFile
END IF

' main program loop starts here.
TopProgram:
CALL SMouse

' reset mouse activity.
IF Mouse.Present THEN
   CALL ClearMouse
END IF

' setup editing screen.
GOSUB RedrawScreen0
CALL SMouse
GOSUB RestorePosition
CALL locatecursor2

' keyboard/mouse input loop starts here.
StartLoop:

' keyboard/mouse input loop.
DO
   ' check for keypress or mouse activity.
   CharInput$ = Nul
   DO
      ' store keyboard buffer.
      CharInput$ = INKEY$
      IF LEN(CharInput$) THEN
         EXIT DO
      END IF

      ' call mouse subroutine.
      CALL MouseDriver2

      ' check left mouse button release.
      IF Mouse.ButtonX THEN
         ' store mouse position.
         Mouse.Row = Mouse.RowX
         Mouse.Column = Mouse.ColumnX
         ' process mouse row/column.
         IF Mouse.Drag = False THEN
            GOSUB MouseButton1
         END IF
         ' init mouse drag pivot byte.
         CopyStart = False
         Time1 = SFalse
         IF Mouse.Row > 3 AND Mouse.Row < 20 THEN
            IF Mouse.Column >= 6 AND Mouse.Column <= 49 THEN
               CopyStart = FilePosition
            END IF
         END IF
      ELSE
         ' check left mouse button drag.
         IF Mouse.Button1 = 2 THEN
            IF Mouse.Row OR Mouse.Column THEN
               Time1 = SFalse
               Mouse.Drag = True
               CALL MouseButton1Drag(0)
            END IF
         ELSE
            ' check left mouse button drag/scroll up.
            IF Mouse.Button1 = 3 THEN
               IF Mouse.Row OR Mouse.Column THEN
                  Mouse.Drag = True
                  CALL MouseButton1Drag(9)
               END IF
            ELSE
               ' check left mouse button drag/scroll down.
               IF Mouse.Button1 = 4 THEN
                  IF Mouse.Row OR Mouse.Column THEN
                     Mouse.Drag = True
                     CALL MouseButton1Drag(10)
                  END IF
               ELSE
                  ' check right/middle mouse button.
                  IF Mouse.Button2 OR Mouse.Button3 THEN
                     GOSUB MouseButton2
                  ELSE
                     ' check mouse position for mouseover.
                     IF Mouse.Row OR Mouse.Column THEN
                        Mouse.Drag = False
                        GOSUB MoveMouse
                     END IF
                  END IF
               END IF
            END IF
         END IF
      END IF

      ' release time slice.
      ' (speeds up mouse in windows).
      R = ReleaseTime
   LOOP

   ' reset mouse activity.
   CALL HMouse
   CALL locatecursor2

   ' get shift state.
   DEF SEG = &H40
   ShiftFlag = PEEK(&H17)
   DEF SEG

   ' store shift state.
   LeftShift = ShiftFlag AND 1 ' left shift.
   RightShift = ShiftFlag AND 2 ' right shift.

   ' check shift flags.
   IF LeftShift OR RightShift THEN
      ' parse the key.
      SELECT CASE LEN(CharInput$)
      CASE 1
         SELECT CASE ASC(CharInput$)
         CASE 124 ' | FirstFile
            GOSUB GotoFirstFile
         CASE 35 ' # LastFile
            GOSUB GotoLastFile
         CASE 33 ' ! = display page/row/column.
            GOSUB DisplayData1
         CASE 34 ' " = display file path.
            GOSUB DisplayPath
         CASE 63 ' ? = display filelength\undos.
            GOSUB DisplayInfo
         CASE 123 ' { = display paste entries.
            GOSUB DisplayPasteUndos
         CASE 126 ' ~ = display last page\markers.
            GOSUB DisplayLengths
         CASE 8, 60 ' Backspace/< file left.
            GOSUB GotoPreviousFile
         CASE 62 ' > file right
            GOSUB GotoNextFile
         CASE 43 ' + = list range of markers.
            IF FileLength THEN
               CALL MultiFileFunction(2)
               CALL Marker(1)
               IF ValidFunction THEN
                  GOSUB RedrawScreen0
               END IF
            END IF
         ' byte hilighting functions.
         CASE 52 ' Shift-Left
            IF FileLength THEN
               CALL MouseButton1Drag(1)
            END IF
         CASE 54 ' Shift-Right
            IF FileLength THEN
               CALL MouseButton1Drag(2)
            END IF
         CASE 56 ' Shift-Up
            IF FileLength THEN
               CALL MouseButton1Drag(3)
            END IF
         CASE 50 ' Shift-Down
            IF FileLength THEN
               CALL MouseButton1Drag(4)
            END IF
         CASE 57 ' Shift-PageUp
            IF FileLength THEN
               CALL MouseButton1Drag(5)
            END IF
         CASE 51 ' Shift-PageDown
            IF FileLength THEN
               CALL MouseButton1Drag(6)
            END IF
         CASE 49 ' Shift-End
            IF FileLength THEN
               CALL MouseButton1Drag(7)
            END IF
         CASE 55 ' Shift-Home
            IF FileLength THEN
               CALL MouseButton1Drag(8)
            END IF
         CASE ELSE ' Unknown key.
            GOSUB HelpLine
         END SELECT
      CASE 2
         SELECT CASE ASC(RIGHT$(CharInput$, 1))
         CASE 15 ' Shift-Tab = Toggle window.
            IF FileLength THEN
               GOSUB TabWindow
            END IF
         ' byte hilighting functions.
         CASE 75 ' Shift-Left
            IF FileLength THEN
               CALL MouseButton1Drag(1)
            END IF
         CASE 77 ' Shift-Right
            IF FileLength THEN
               CALL MouseButton1Drag(2)
            END IF
         CASE 72 ' Shift-Up
            IF FileLength THEN
               CALL MouseButton1Drag(3)
            END IF
         CASE 80 ' Shift-Down
            IF FileLength THEN
               CALL MouseButton1Drag(4)
            END IF
         CASE 73 ' Shift-PageUp
            IF FileLength THEN
               CALL MouseButton1Drag(5)
            END IF
         CASE 81 ' Shift-PageDown
            IF FileLength THEN
               CALL MouseButton1Drag(6)
            END IF
         CASE 79 ' Shift-End
            IF FileLength THEN
               CALL MouseButton1Drag(7)
            END IF
         CASE 71 ' Shift-Home
            IF FileLength THEN
               CALL MouseButton1Drag(8)
            END IF
         CASE ELSE ' Unknown key.
            GOSUB HelpLine
         END SELECT
      END SELECT
   ELSE
      ' parse the key.
      SELECT CASE LEN(CharInput$)
      CASE 1 ' single ascii key.
         SELECT CASE ASC(CharInput$)
         CASE 5 ' Control-E = TestError
            IF Debug THEN
               CALL DisplayX
               COLORf White
               PRINTf "Test on."
               CALL locatecursor2
               Check.Disk = 0
               Value% = SQR(-1) ' test syntax error.
               CALL DisplayX
               COLORf White
               PRINTf "Test off."
               CALL locatecursor2
            END IF
         CASE 9 ' Tab = Toggle window.
            IF FileLength THEN
               GOSUB TabWindow
            END IF
         CASE 13 ' Enter = Change byte value.
            IF FileLocked THEN
               CALL DisplayX
               CALL lockedfile
            ELSE
               IF FileLength = 0 THEN
                  Byte2 = 0
                  GOSUB AppendIt
               END IF
               IF FileLength THEN
                  GOSUB ChangeHexValue
               END IF
            END IF
         CASE 27 ' Escape = Start drop down menu.
            IF CurrentMenu <> 1 THEN
               ' remove hilight menu tab.
               COLORf2 White, 0
               CALL DisplayTab(CurrentMenu)
            END IF
            CurrentMenu = 1
            CurrentMenuSelection = 1
            GOTO Menu2
         CASE 59 ' ; = display hilight range.
            GOSUB DisplayCopy
         CASE 49 ' 1 = Add marker.
            IF FileLength THEN
               CALL Marker(2)
            END IF
         CASE 50 ' 2 = Add marker value.
            IF FileLength THEN
               CALL Marker(3)
            END IF
         CASE 51 ' 3 = Delete marker.
            IF FileLength THEN
               CALL Marker(4)
            END IF
         CASE 52 ' 4 = Delete specified marker.
            IF FileLength THEN
               CALL Marker(5)
            END IF
         CASE 53 ' 5 = Jump to previous marker.
            IF FileLength THEN
               CALL Marker(6)
            END IF
         CASE 54 ' 6 = Jump to current marker.
            IF FileLength THEN
               CALL Marker(7)
            END IF
         CASE 55 ' 7 = Jump to next marker.
            IF FileLength THEN
               CALL Marker(8)
            END IF
         CASE 56 ' 8 = Jump to specified marker.
            IF FileLength THEN
               CALL Marker(9)
            END IF
         CASE 57 ' 9 = List all markers.
            IF FileLength THEN
               CALL MultiFileFunction(2)
               CALL Marker(10)
               GOSUB RedrawScreen0
            END IF
         CASE 48 ' 0 = List specified markers.
            IF FileLength THEN
               CALL Marker(11)
            END IF
         CASE 45 ' - = List marker of value.
            IF FileLength THEN
               CALL MultiFileFunction(2)
               CALL Marker(12)
               GOSUB RedrawScreen0
            END IF
         CASE 61 ' = = Delete range of markers.
            IF FileLength THEN
               CALL Marker(13)
            END IF
         CASE 91 ' [ = Delete all markers.
            IF FileLength THEN
               CALL Marker(14)
            END IF
         CASE 93 ' ] = Delete markers of specified values.
            IF FileLength THEN
               CALL Marker(15)
            END IF
         ' toggle debug.
         CASE 4 ' Control-D = Toggle debug variable.
            GOSUB ToggleDebug
         CASE 6 ' Control-F = Toggle debug2 variable.
            GOSUB ToggleDebug2
         CASE 7 ' Control-G = Toggle debug3 variable.
            GOSUB ToggleDebug3
         ' clipboard functions.
         CASE 3 ' Control-C = Copy paste area.
            IF FileLength THEN
               CALL Clipboard2(1)
            END IF
         CASE 22 ' Control-V = Paste area.
            IF FileLength THEN
               CALL Clipboard2(2)
            END IF
         CASE 24 ' Control-X = Undo current paste.
            IF FileLength THEN
               CALL Clipboard2(3)
            END IF
         CASE 26 ' Control-Z = Undo all pastes.
            IF FileLength THEN
               CALL Clipboard2(4)
            END IF
         ' remaining control functions.
         CASE 19 ' Control-S = Windows screen saver.
            GOSUB SaveShell
         CASE 20 ' Control-T = Toggle search string storage variable.
            GOSUB ToggleSearch
         ' remaining functions.
         CASE 32 ' Space = Display author and help key.
            GOSUB DisplayAuthor
         CASE 8, 127 ' Backspace/Control-BackSpace = Goto previous file.
            GOSUB GotoPreviousFile
         CASE ELSE ' Unknown key.
            GOSUB HelpLine2
         END SELECT
      CASE 2 ' extended ascii key.
         SELECT CASE ASC(RIGHT$(CharInput$, 1))
         CASE 0 ' Control-Break = Display Break/Exit Program.
            ' Note: Control-Break places word 0000h in keyboard buffer,
            '   invokes INT 1Bh, and sets flag 80h at memory 0040h:0071h
            IF Debug THEN
               CALL DisplayX
               COLORf White
               PRINTf "*break*"
               CALL locatecursor2
            ELSE
               EXIT DO
            END IF
         CASE 94, 130 ' Control-F1/Alt-- = Display Date/Time
            GOSUB DisplayData2
         CASE 95, 131 ' Control-F2/Alt-= = Display loaded files.
            GOSUB DisplayData3
         CASE 30 ' Alt-A = Append multiple bytes.
            IF FileLocked THEN
               CALL DisplayX
               CALL lockedfile
            ELSE
               GOSUB AppendByte
            END IF
         CASE 48 ' Alt-B = Append multiple null bytes.
            IF FileLocked THEN
               CALL DisplayX
               CALL lockedfile
            ELSE
               GOSUB AppendNullBytes
            END IF
         CASE 46 ' Alt-C = ANSI Chart.
            GOSUB ANSIChart
         CASE 32 ' Alt-D = HEX screen dump.
            IF FileLength THEN
               GOSUB DumpScreen
            END IF
         CASE 18 ' Alt-E = HEX file dump.
            IF FileLength THEN
               GOSUB DumpHEXFile
            END IF
         CASE 33, 148 ' Alt-F/Control-Tab = Start drop down menu.
            IF CurrentMenu <> 1 THEN
               ' remove hilight menu tab.
               COLORf2 White, 0
               CALL DisplayTab(CurrentMenu)
            END IF
            CurrentMenu = 1
            CurrentMenuSelection = 1
            GOTO Menu2
         CASE 34 ' Alt-G = Append multiple specified bytes.
            IF FileLocked THEN
               CALL DisplayX
               CALL lockedfile
            ELSE
               GOSUB AppendAnyByte
            END IF
         CASE 35 ' Alt-H = HEX chart.
            GOSUB HEXChart
         CASE 36 ' Alt-J = Jump to byte.
            IF FileLength THEN
               GOSUB JumpByte
            END IF
         CASE 38 ' Alt-L = Jump to page.
            IF FileLength THEN
               GOSUB JumpPage
            END IF
         CASE 50 ' Alt-M = Append ASCII string.
            IF FileLocked THEN
               CALL DisplayX
               CALL lockedfile
            ELSE
               GOSUB AppendASCII
            END IF
         CASE 49, 66, 101 ' Alt-N/F8/Control-F8 = Load new file.
            IF NumberFiles = MaxFiles THEN
               StatusMessage = "Maximum of" + STR$(MaxFiles) + " files open."
               CALL displaystatus2
            ELSE
               CALL MultiFileFunction(2)
               GOSUB NewFile
               GOTO BeginFile
            END IF
         CASE 24 ' Alt-O = Print help screens.
            GOSUB PrintHelp
         CASE 25 ' Alt-P = HEX screen print.
            IF FileLength THEN
               GOSUB PrintScreen
            END IF
         CASE 16 ' Alt-Q = Display help screens.
            GOSUB DisplayHelp
         CASE 19 ' Alt-R = Redraw screen.
            CALL MultiFileFunction(2)
            GOSUB RedrawScreen0
         CASE 20 ' Alt-T = HEX file print.
            IF FileLength THEN
               GOSUB PrintFile
            END IF
         CASE 22 ' Alt-U = Undo last byte change.
            IF FileLength THEN
               GOSUB UndoByte
            END IF
         CASE 17, 67, 102 ' Alt-W/F9/Control-F9 = Close current file.
            GOSUB CloseCurrentFile
            IF NumberFiles = False THEN
               GOSUB NewFile
               GOTO BeginFile
            END IF
         CASE 45 ' Alt-X = Exit program.
            EXIT DO ' exit program.
         CASE 21, 63 ' Alt-Y/F5 = Toggle right window.
            IF FileLength THEN
               GOSUB ToggleRight
            END IF
         CASE 44 ' Alt-Z = Undo all byte changes.
            IF FileLength THEN
               GOSUB UndoAll
            END IF
         CASE 129 ' Alt-0 = Close all files/Return to File Menu Box.
            GOSUB CloseAllFiles
            GOSUB ResetAllAttributes
            GOSUB NewFile
            NumberFiles = False
            GOTO BeginFile
         CASE 59, 141 ' F1/Control-Up = Toggle filename display.
            FileDisplay = NOT FileDisplay
            IF FileDisplay = False THEN
               CALL DisplayFileTitle
            ELSE
               GOSUB DisplayScreen3
            END IF
         CASE 60, 145 ' F2/Control-Down = Toggle bottom row help keys.
            ScreenRow = NOT ScreenRow
            CALL MultiFileFunction(2)
            GOSUB RedrawScreen0
         CASE 37, 61 ' Alt-K/F3 = Search ascii string.
            IF FileLength THEN
               GOSUB SearchASCII
            END IF
         ' Alt-S/F4/Control-F4 = Search multiple bytes.
         CASE 31, 62, 97
            IF FileLength THEN
               GOSUB SearchBytes
            END IF
         ' Alt-Right/F6/Control-F6 = Goto next file.
         CASE 157, 64, 99
            GOSUB GotoNextFile
         ' Alt-Left/Alt-BackSpace/F7/Control-F7 = Goto previous file.
         CASE 155, 14, 65, 100
            GOSUB GotoPreviousFile
         ' Alt-V/Alt-F10/F10 = View files.
         CASE 47, 103, 68
            CALL ViewFiles(Var)
            IF Var > False THEN
               NextFile = Var
               GOSUB SearchFile3
            END IF
            CALL displayfilename
         CASE 96 ' Control-F3 = Continue search.
            IF FileLength THEN
               GOSUB ContinueSearch
            END IF
         ' F11/Control-F11/Alt-F11 = Drop to DOS.
         CASE 133, 137, 139
            GOSUB DOSshell
         ' Alt-I/F12/Control-F12/Alt-F12 = DOS Command.
         CASE 23, 134, 138, 140
            GOSUB DOScommand
         ' Control-Keypad-5/Control-F5 = Select Specific Filename.
         CASE 143, 98
            GOSUB SearchFile2
         CASE 76, 113 ' Keypad-5/Alt-F10 = Hex/Dec calculator.
            GOSUB HEXCalculator
         CASE 104 TO 112 ' Alt-F1 to Alt-F9 = Select File.
            NextFile = ASC(RIGHT$(CharInput$, 1)) - 103
            GOSUB SearchFile3
         CASE 120 TO 128 ' Alt-1 to Alt-9 = Select file.
            NextFile = ASC(RIGHT$(CharInput$, 1)) - 119
            GOSUB SearchFile3
         CASE 82 ' Insert = Change string value.
            IF FileLocked THEN
               CALL DisplayX
               CALL lockedfile
            ELSE
               GOSUB ChangeASCIIValues
            END IF
         CASE 83 ' Delete = Change HEX string.
            IF FileLocked THEN
               CALL DisplayX
               CALL lockedfile
            ELSE
               GOSUB ChangeHEXValues
            END IF
         CASE 72 ' Up
            IF FileLength THEN
               GOSUB CursorUp
            END IF
         CASE 80 ' Down
            IF FileLength THEN
               GOSUB CursorDown
            END IF
         CASE 75 ' Left
            IF FileLength THEN
               GOSUB CursorLeft
            END IF
         CASE 77 ' Right
            IF FileLength THEN
               GOSUB CursorRight
            END IF
         CASE 152, 73, 132 ' Alt-Up/PageUp/Control-PageUp
            IF FileLength THEN
               GOSUB PageUp
            END IF
         CASE 160, 81, 118 ' Alt-Down/PageDown/Control-PageDown
            IF FileLength THEN
               GOSUB PageDown
            END IF
         CASE 79 ' End
            IF FileLength THEN
               GOSUB EndKey
            END IF
         CASE 71 ' Home
            IF FileLength THEN
               GOSUB HomeKey
            END IF
         CASE 142, 74, 115 ' Control-Keypad-/Alt-Keypad-/Control-Left
            IF FileLength THEN
               GOSUB CtrlLeftKey
            END IF
         CASE 144, 78, 116 ' Control-Keypad+/Alt-Keypad+/Control-Right
            IF FileLength THEN
               GOSUB CtrlRightKey
            END IF
         CASE 117 ' Control-End
            IF FileLength THEN
               GOSUB CtrlEndKey
            END IF
         CASE 119 ' Control-Home
            IF FileLength THEN
               GOSUB CtrlHomeKey
            END IF
         CASE ELSE ' Unknown key.
            GOSUB HelpLine3
         END SELECT
      END SELECT
   END IF
   ' reset mouse activity.
   CALL SMouse
   GOSUB RestorePosition
   CALL locatecursor2
LOOP

' stop program.
StopLabel:

' clear hilighted byte.
CALL ClearPageByte
GOSUB MoveOldPosition

' clear status areas.
CALL clearstatus
GOSUB DisplayScreen3

' display goodbye message.
COLORf2 White, 0
LOCATEf 2, 4, 0
Z$ = "You have been using a public domain program written by "
Z$ = Z$ + Author$ + ".."
Z$ = LEFT$(Z$, 74)
PRINTf Z$
LOCATEf 2, 4, 0
COLORf2 Plain, 0

' clear file menu tab.
IF CurrentMenu THEN
   COLORf2 White, 0
   CALL DisplayTab(CurrentMenu)
   COLORf2 Plain, 0
END IF

' remove bottom lines of screen.
FOR Var = 21 TO 24
   LOCATEf Var, 1, 0
   PRINTf SPACE$(78)
NEXT
LOCATEf 21, 1, 0
COLORf Magenta
PRINTf " " + CHR$(LLcorner) + STRING$(75, Hline) + CHR$(LRcorner)
LOCATEf 22, 1, 0
Colorf2 Plain, 0
Print " ";

' goto end of program.
GOTO Stoploop
END

' load filespec.
LoadMultipleFile:
 ' check max files.
 IF NumberFiles = MaxFiles THEN
    RETURN
 END IF

 ' trim filename.
 Filename = Var$
 Filename = RTRIM$(Filename)
 Filename = LTRIM$(Filename)

 ' check long filename in quotes.
 IF LEFT$(Filename, 1) = Quote THEN
    IF RIGHT$(Filename, 1) = Quote THEN
       Filename = MID$(Filename, 2)
       Filename = LEFT$(Filename, LEN(Filename) - 1)
    END IF
 END IF

 ' check wildcard characters.
 IF INSTR(Filename, "?") OR INSTR(Filename, "*") THEN
    ' store filename drive/directory
    D$ = Nul
    C$ = Nul
    N$ = Nul
    X$ = Nul
    ' get netpath
    IF LEFT$(Filename, 2) = "\\" THEN
       Net1 = INSTR(3, Filename, "\")
       IF Net1 THEN
          Net2 = INSTR(Net1 + 1, Filename, "\")
          IF Net2 THEN
             N$ = LEFT$(Filename, Net2 - 1)
             Filename = MID$(Filename, Net2)
          END IF
       END IF
    END IF
    ' get directory
    FOR Var = LEN(Filename) TO 1 STEP -1
       IF MID$(Filename, Var, 1) = "\" THEN
          D$ = LEFT$(Filename, Var)
          X$ = MID$(Filename, Var + 1)
          EXIT FOR
       END IF
    NEXT
    ' restore filename
    IF N$ <> Nul THEN
       Filename = N$ + D$ + X$
    END IF
    ' get drive
    IF MID$(Filename, 2, 1) = ":" THEN
       C$ = LEFT$(Filename, 2)
       D$ = MID$(D$, 3)
    END IF
    ' clear filename array.
    FOR VarZ = 1 TO MaxFiles
       TempFiles(VarZ) = Nul
    NEXT
    ' store filename array.
    VarZ = False
    IF Windows.Detected THEN
       X$ = DIRz$(Filename)
    ELSE
       X$ = DIR$(Filename)
    END IF
    DO
       IF LEN(X$) THEN
          IF VarZ < MaxFiles THEN
             VarZ = VarZ + 1
             IF N$ = Nul THEN
                TempFiles(VarZ) = C$ + D$ + X$
             ELSE
                IF D$ = Nul THEN
                   TempFiles(VarZ) = N$ + "\" + X$
                ELSE
                   TempFiles(VarZ) = N$ + "\" + D$ + X$
                END IF
             END IF
          ELSE
             EXIT DO
          END IF
       ELSE
          EXIT DO
       END IF
       IF Windows.Detected THEN
          X$ = DIRz$(Nul)
       ELSE
          X$ = DIR$
       END IF
    LOOP
    IF Windows.Detected THEN
       ' close long filename search
       InregsX.AX = &H71A1
       InregsX.BX = Wfile.Handle
       CALL InterruptX(&H21, InregsX, OutregsX)
    END IF

    ' load filename array.
    FOR VarZ = 1 TO MaxFiles
       IF RTRIM$(TempFiles(VarZ)) <> "" THEN
          ' check max files.
          IF NumberFiles = MaxFiles THEN
             RETURN
          END IF
          Filename = RTRIM$(TempFiles(VarZ))
          GOSUB LoadNewFile
       END IF
    NEXT
 ELSE
    ' load single file.
    GOSUB LoadNewFile
 END IF
 RETURN

' loads filename.
LoadNewFile:
 ' check max files.
 IF NumberFiles = MaxFiles THEN
    RETURN
 END IF

 ' store filename.
 ASCIIZ = Filename + CHR$(0)

 ' open/create file.
 GOSUB OpenFile

 ' check open error.
 IF ValidFile THEN

    ' get netpath
    GOSUB ParseNetPath

    ' conanicalize filename.
    GOSUB ShortFilename

    ' get length of file.
    CALL GetFileLength(Var)

    ' check filelength.
    IF Var THEN
       ' close handle.
       GOSUB CloseFile

       ' restore current file.
       GOSUB RestoreCurrentFile
       RETURN
    END IF

    ' get exclusion list.
    GOSUB CheckFile

    ' check excluded file.
    IF VarP THEN
       ' close handle.
       GOSUB CloseFile

       ' restore current file.
       GOSUB RestoreCurrentFile
       RETURN
    END IF

    ' initialize new file.
    GOSUB InitFile

    ' init paste files.
    CALL InitPasteFiles(VarQ)

    ' check bad paste filename.
    IF VarQ THEN
       GOTO Error.Exit
    END IF
 END IF
 RETURN

' initialize new file.
InitFile:
 NumberFiles = NumberFiles + 1
 CurrentFile = NumberFiles
 CALL MultiFileFunction(1)
 GOSUB RestoreCurrentFile
 RETURN

' check file exclusion list.
CheckFile:
 VarP = False
 Filename2$ = ExcludeFile
 Var = INSTR(Filename2$, "?")
 IF Var THEN
    MID$(Filename2$, Var, 1) = MID$(STR$(Process.Number), 2)
 END IF
 ASCIIZ3 = ASCIIZ
 ASCIIZ = Filename2$ + CHR$(0)
 CALL GetShortFilename(Short.Filename$)
 ASCIIZ = ASCIIZ3
 Filename2$ = Short.Filename$
 IF DIR$(Filename2$) <> Nul THEN
    Check.Disk = True
    Disk.Ready = False
    CLOSE #2
    OPEN Filename2$ FOR INPUT AS #2
    Check.Disk = False
    IF Disk.Ready THEN
       RETURN
    END IF
    Y2$ = Filename
    Y2$ = UCASE$(Y2$)
    FOR Y = LEN(Y2$) TO 1 STEP -1
       IF MID$(Y2$, Y, 1) = "\" THEN
          Y2$ = MID$(Y2$, Y + 1)
          EXIT FOR
       END IF
    NEXT
    DO UNTIL EOF(2)
       LINE INPUT #2, Y1$
       Y1$ = LTRIM$(Y1$)
       Y1$ = RTRIM$(Y1$)
       Y1$ = UCASE$(Y1$)
       IF LEFT$(Y1$, 1) = Quote THEN
          Y1$ = MID$(Y1$, 2)
          IF RIGHT$(Y1$, 1) = Quote THEN
             Y1$ = LEFT$(Y1$, LEN(Y1$) - 1)
          END IF
       END IF
       Y1$ = LTRIM$(Y1$)
       Y1$ = RTRIM$(Y1$)
       IF LEN(Y1$) THEN
          IF INSTR(Y1$, " ") = False THEN
             ' check excluded file in filename.
             CALL CheckExcluded(Y1$, Y2$, Exclude.File%)
             IF Exclude.File% THEN
                VarP = True
                EXIT DO
             END IF
          END IF
       END IF
    LOOP
 END IF
 RETURN

' search specific file name.
'   allows ? and * characters.
SearchFile2:
 CALL clearstatus
 PRINTf "Enter filename?"
 Var1$ = LineInput$(2, 20, 10)
 IF LEN(Var1$) THEN
    FOR NextFile = CurrentFile + 1 TO NumberFiles
       Var2$ = RTRIM$(File(NextFile).Filename)
       IF LEN(Var2$) THEN
          CALL InstrSUB(VarX, Var2$, Var1$)
          IF VarX THEN
             GOTO SearchFile3
          END IF
       END IF
    NEXT
    FOR NextFile = 1 TO CurrentFile
       Var2$ = RTRIM$(File(NextFile).Filename)
       IF LEN(Var2$) THEN
          CALL InstrSUB(VarX, Var2$, Var1$)
          IF VarX THEN
             GOTO SearchFile3
          END IF
       END IF
    NEXT
 END IF
 StatusMessage = "File not found."
 CALL displaystatus2
 RETURN

' select specific file number.
SearchFile3:
 IF NextFile <= NumberFiles THEN
    CALL MultiFileFunction(2)
    CurrentFile = NextFile
    GOSUB RestoreCurrentFile
    GOSUB RedrawScreen2
 END IF
 RETURN

' goto next file.
GotoNextFile:
 IF NumberFiles = 1 THEN
    RETURN
 END IF
 CALL MultiFileFunction(2)
 CurrentFile = CurrentFile + 1
 IF CurrentFile > NumberFiles THEN
    CurrentFile = 1
 END IF
 GOSUB RestoreCurrentFile
 GOSUB RedrawScreen2
 RETURN

' goto previous file.
GotoPreviousFile:
 IF NumberFiles = 1 THEN
    RETURN
 END IF
 CALL MultiFileFunction(2)
 CurrentFile = CurrentFile - 1
 IF CurrentFile = False THEN
    CurrentFile = NumberFiles
 END IF
 GOSUB RestoreCurrentFile
 GOSUB RedrawScreen2
 RETURN

' goto first file.
GotoFirstFile:
 IF CurrentFile = 1 THEN
    RETURN
 END IF
 CALL MultiFileFunction(2)
 CurrentFile = 1
 GOSUB RestoreCurrentFile
 GOSUB RedrawScreen2
 RETURN

' goto last file.
GotoLastFile:
 IF CurrentFile = NumberFiles THEN
    RETURN
 END IF
 CALL MultiFileFunction(2)
 CurrentFile = NumberFiles
 GOSUB RestoreCurrentFile
 GOSUB RedrawScreen2
 RETURN

' close current file.
CloseCurrentFile:
 GOSUB CloseFile
 GOSUB ResetAttribute
 GOSUB PackFileArray
 RETURN

' pack file arrays and structures.
PackFileArray:
 ' pack file array.
 FOR Var = CurrentFile TO MaxFiles - 1
    File(Var) = File(Var + 1)
 NEXT

 ' pack paste files.
 CALL Pack.Files1

 ' pack undo file.
 CALL Pack.Arrays1

 ' pack marker file.
 CALL Pack.Arrays2

 ' reduce files by one.
 NumberFiles = NumberFiles - 1

 ' check remaining files.
 IF NumberFiles > False THEN
    ' reset current file.
    IF CurrentFile > NumberFiles THEN
       CurrentFile = NumberFiles
    END IF

    ' restore current file.
    GOSUB RestoreCurrentFile
    GOSUB RedrawScreen2
 END IF
 RETURN

' prepare for new file.
NewFile:
 ' reset window.
 CLS
 LOCATEf 1, 1, 1
 PRINTf "Loading.."
 CurrentWindow = False
 CurrentWindow2 = False
 ScreenDrawn = False
 LOCATEf 24, 1, 1
 COLORf Plain
 RETURN

' restore current file variables.
RestoreCurrentFile:
 IF NumberFiles = 0 THEN
    RETURN
 END IF
 CALL MultiFileFunction(3)
 CALL OpenPasteFiles(VarQ)
 IF VarQ THEN
    GOTO Error.Exit
 END IF
 CALL DisplayScreen
 IF FileDisplay = False THEN
    CALL DisplayFileTitle
 ELSE
    GOSUB DisplayScreen3
 END IF
 RETURN

' display command line instructions.
BootUsage:
 CALL DisplayBootUsage
 COLORf Plain
 GOTO StopProgram

' display ANSI charts.
ANSIChart:
 CALL DisplayChart(1)
 GOTO EndChart
' display HEX charts.
HEXChart:
 CALL DisplayChart(2)
 GOTO EndChart
' display HEX calculator.
HEXCalculator:
 CALL HexCalc
EndChart:
 CALL MultiFileFunction(2)
 GOSUB RedrawScreen0
 RETURN

' toggles state of search string storage variable.
ToggleSearch:
 StoreSearchMulti = NOT StoreSearchMulti
 CALL DisplayX
 COLORf White
 IF StoreSearchMulti THEN
    PRINTf "Store search string on."
 ELSE
    PRINTf "Store search string off."
 END IF
 CALL locatecursor2
 RETURN

' toggles state of debug variable.
ToggleDebug:
 Debug = NOT Debug
 CALL DisplayX
 COLORf White
 IF Debug THEN
    PRINTf "Debug on."
 ELSE
    PRINTf "Debug off."
 END IF
 CALL locatecursor2
 RETURN

' toggles state of debug2 variable.
ToggleDebug2:
 Debug2 = NOT Debug2
 CALL DisplayX
 COLORf White
 IF Debug2 THEN
    PRINTf "Debug2 on."
 ELSE
    PRINTf "Debug2 off."
 END IF
 CALL locatecursor2
 RETURN

' toggles state of debug3 variable.
ToggleDebug3:
 Debug3 = NOT Debug3
 CALL DisplayX
 COLORf White
 IF Debug3 THEN
    PRINTf "Debug3 on."
 ELSE
    PRINTf "Debug3 off."
 END IF
 CALL locatecursor2
 RETURN

' function for unknown ascii key.
HelpLine2:
 CALL DisplayX
 COLORf White
 IF Debug THEN
    PRINTf MID$(STR$(ASC(CharInput$)), 2) + ": "
 END IF
 GOTO HelpLineX

' function for unknown scan key.
HelpLine3:
 CALL DisplayX
 COLORf White
 IF Debug THEN
    PRINTf MID$(STR$(ASC(RIGHT$(CharInput$, 1))), 2) + ": "
 END IF
 GOTO HelpLineX

' function for unknown key.
HelpLine:
 CALL DisplayX
 COLORf White
HelpLineX:
 PRINTf "Type Alt-Q for Help, Alt-O to print Help."
 CALL locatecursor2
 RETURN

' displays help screens.
DisplayHelp:
 CALL HMouse
 CALL MultiFileFunction(2)
 CALL HelpScreen
 GOSUB RedrawScreen0
 CALL SMouse
 RETURN

' function for ? key.
DisplayInfo:
 CALL DisplayX
 COLORf White
 IF FileLength = False THEN
    CALL lockedfile
    RETURN
 END IF
 Attr$ = Nul
 IF (FileAttribute AND 1) = 1 THEN
    Attr$ = Attr$ + "o"
 END IF
 IF (FileAttribute AND 2) = 2 THEN
    Attr$ = Attr$ + "h"
 END IF
 IF (FileAttribute AND 4) = 4 THEN
    Attr$ = Attr$ + "s"
 END IF
 IF (FileAttribute AND 32) = 32 THEN
    Attr$ = Attr$ + "a"
 END IF
 IF LEN(Attr$) THEN
    PRINTf "(" + Attr$ + ") "
 END IF
 IF CurrentWindow2 = False THEN
    PRINTf "(Length: " + FormatX$(FileLength, 1) + ") "
 ELSE
    PRINTf "(Length: " + RIGHT$("00000000" + HEX$(FileLength - 1), 8) + "H) "
 END IF
 PRINTf "(Undos: " + FormatX$(CDBL(CurrentUndo), 1) + ") "
 CALL locatecursor2
 RETURN

' function for space key.
DisplayAuthor:
 CALL DisplayX
 COLORf White
 PRINTf "Author: " + Author$ + " Type Alt-Q for help."
 CALL locatecursor2
 RETURN

' function for ! key.
DisplayData1:
 CALL DisplayX
 COLORf White
 IF FileLength = False THEN
    CALL lockedfile
    RETURN
 END IF
 TempPosition3 = FilePosition
 CALL FormatPosition1
 CALL locatecursor2
 RETURN

' function for Alt-- key.
DisplayData2:
 CALL DisplayX
 COLORf White
 PRINTf Format$(Now, "ddd dd mmm yyyy\, hh:mma/p")
 CALL locatecursor2
 RETURN

' function for Alt-= key.
DisplayData3:
 CALL DisplayX
 COLORf White
 PRINTf "Loaded Files:" + STR$(NumberFiles) + " "
 PRINTf "Last Drive: " + CHR$(Last.Drive + 64) + ":"
 IF RTRIM$(RAMdisk) <> "" THEN
    PRINTf " RAM disk: " + RAMdisk + ":"
 END IF
 CALL locatecursor2
 RETURN

' function for ~ key.
DisplayLengths:
 CALL DisplayX
 COLORf White
 IF FileLength = False THEN
    CALL lockedfile
    RETURN
 END IF
 FilePage2 = INT((FileLength - 1) / 320) + 1
 IF CurrentWindow2 = False THEN
    PRINTf "(Last page: " + FormatX$(CDBL(FilePage2 - 1), 1) + ") "
 ELSE
    PRINTf "(Last page: " + RIGHT$("00000000" + HEX$(FilePage2 - 1), 8) + "H) "
 END IF
 ' count markers
 Var1! = SFalse
 FOR Var2! = 1! TO MarkerCount
    IF Markers#(Var2!) > 0# THEN
       Var1! = Var1! + 1!
    END IF
 NEXT
 PRINTf "(Markers: " + FormatX$(CDBL(Var1!), 1) + ")"
 CALL locatecursor2
 RETURN

' function for ; key.
DisplayCopy:
 CALL DisplayX
 COLORf White
 IF FileLength = False THEN
    CALL lockedfile
    RETURN
 END IF
 IF CurrentWindow2 = False THEN
    PRINTf "(Copy position: "
    PRINTf FormatX$(CDBL(CopyPositionStart), 1) + " - "
    PRINTf FormatX$(CDBL(CopyPositionEnd), 1) + ") "
 ELSE
    PRINTf "(CopyPosition: "
    IF CopyPositionStart = False AND CopyPositionEnd = False THEN
       PRINTf "00000000H - 00000000H) "
    ELSE
       PRINTf RIGHT$("00000000" + HEX$(CopyPositionStart - 1), 8) + "H - "
       PRINTf RIGHT$("00000000" + HEX$(CopyPositionEnd - 1), 8) + "H) "
    END IF
 END IF
 CALL locatecursor2
 RETURN

' { = display paste undos.
DisplayPasteUndos:
 CALL DisplayX
 COLORf White
 IF FileLength = False THEN
    CALL lockedfile
    RETURN
 END IF
 ByteEntries = 0
 IF LOF(5) > 0 THEN
    GET #5, 1, CopyByte
    ByteEntries = ASC(CopyByte) * 16 ^ 2
    GET #5, 2, CopyByte
    ByteEntries = ByteEntries + ASC(CopyByte)
 END IF
 PRINTf FormatX$(CDBL(ByteEntries), 1) + " "
 IF ByteEntries = 1 THEN
    PRINTf "paste undo stored."
 ELSE
    PRINTf "paste undos stored."
 END IF
 CALL locatecursor2
 RETURN

' " = display path of file being edited.
DisplayPath:
 CALL clearstatus
 N$ = RTRIM$(CurrentNetPath)
 Z$ = RTRIM$(ShortFilename)
 IF N$ <> Nul THEN
    IF MID$(Z$, 2, 1) = ":" THEN
       Z$ = MID$(Z$, 3)
    END IF
 END IF
 CALL Deconcatenate(N$, Z$, 64)
 Var = LEN(Z$)
 DO
    Var = Var - 1
    IF Var = False THEN
       Z$ = Nul
       EXIT DO
    END IF
    IF MID$(Z$, Var, 1) = "\" THEN
       Z$ = LEFT$(Z$, Var)
       EXIT DO
    END IF
 LOOP
 IF Z$ = Nul THEN
    Z$ = "\"
 END IF
 PRINTf "Editing path: " + Z$ + " "
 FileDisplay2 = True
 CALL locatecursor2
 RETURN

' redraw editing screen
RedrawScreen0:
 ScreenDrawn = True
 GOSUB RestoreCurrentFile
 GOTO RedrawScreen2
RedrawScreen1:
 ScreenDrawn = True
 CALL DisplayScreen
RedrawScreen2:
 CALL displayfilename
 CALL DisplayHexPage
 CALL displaypagebyte
 IF FileDisplay = False THEN
    CALL DisplayFileTitle
 ELSE
    GOSUB DisplayScreen3
 END IF
 CALL RestoreHilightBytes
 RETURN

' display filename border area.
DisplayScreen3:
 CALL DisplayScreen3X
 CALL locatecursor2
 RETURN

' search file for ascii string.
SearchASCII:
 CALL SearchFile(0, FoundString)
 GOTO FinishSearch
' search file for hex string.
SearchBytes:
 CALL SearchFile(1, FoundString)
 GOTO FinishSearch
' continue previous search.
ContinueSearch:
 IF RTRIM$(StoreSearchString) = Nul THEN
    StatusMessage = "No search string specified."
    CALL displaystatus2
    RETURN
 END IF
 CALL SearchFile(2, FoundString)
' display search results.
FinishSearch:
 ' check byte comparison.
 IF FoundString = True THEN
    FilePosition = SearchBytePosition
    CALL CalculatePosition1
    IF LastPage <> FilePage THEN
       CALL DisplayHexPage
    END IF
 END IF
 CALL displaypagebyte
 IF FoundString = False THEN
    StatusMessage = "String not found."
    CALL displaystatus2
    RETURN
 END IF
 CALL displayfilename
 RETURN

' dump screen of current page of hex/ascii values to file.
DumpScreen:
 ' open dumpfile.
 GOSUB OpenDumpFile
 IF ValidFile = False THEN
    RETURN
 END IF
 ' write screen.
 CALL DumpSub(1)
 Var = 1
 GOSUB DumpIt
 RETURN

' dump file of current hex/ascii values to file.
DumpHEXFile:
 ' open dumpfile.
 GOSUB OpenDumpFile
 IF ValidFile = False THEN
    RETURN
 END IF
 ' write file.
 CALL DumpSub(2)
 Var = 3
 IF FileDumped THEN
    Var = 2
 END IF
 GOSUB DumpIt
 RETURN

' display dump info.
DumpIt:
 ' redisplay page.
 CALL displaypagebyte
 ' conanicalize filename.
 Filename2$ = Conanicalize$(DumpFile)
 Var2 = INSTR(Filename2$, "?")
 IF Var2 THEN
    MID$(Filename2$, Var2, 1) = MID$(STR$(Process.Number), 2)
 END IF
 ' display filename.
 SELECT CASE Var
 CASE 1
    StatusMessage = "Screen dumped to: '" + Filename2$ + "'."
 CASE 2
    StatusMessage = "File dumped to: '" + Filename2$ + "'."
 CASE 3
    StatusMessage = "Partial file dumped to: '" + Filename2$ + "'."
 END SELECT
 CALL DisplayStatus1
 RETURN

' open dumpfile.
OpenDumpFile:
 Column = False
 ColumnSpace = False
 HexLine$ = Nul
 HexLine2$ = Nul

 CLOSE #1
 ErrorTrap = False
 ValidFile = True

 ' store filename.
 Filename2$ = DumpFile
 Var = INSTR(Filename2$, "?")
 IF Var THEN
    MID$(Filename2$, Var, 1) = MID$(STR$(Process.Number), 2)
 END IF

 ' test file exists or could exist.
 IF TestFile(Filename2$) = 0 THEN
    StatusMessage = "Error opening dump filename."
    CALL DisplayStatus1
    ValidFile = False
    RETURN
 END IF

 ' ambiguate filename.
 IF Windows.Detected THEN
    ' store filenames.
    ASCIIZ3 = ASCIIZ
    ASCIIZ = Filename2$ + CHR$(0)

    ' check file exists.
    IF DIRz$(Filename2$) = Nul THEN

       ' create new file.
       InregsX.AX = &H716C
       InregsX.BX = &H2
       InregsX.CX = &H0
       InregsX.DX = &H10
       InregsX.DS = VARSEG(ASCIIZ)
       InregsX.SI = VARPTR(ASCIIZ)
       InregsX.DI = &H1
       CALL InterruptX(&H21, InregsX, OutregsX)
       'Handle = OutregsX.AX
    END IF

    ' close long filename search
    InregsX.AX = &H71A1
    InregsX.BX = Wfile.Handle
    CALL InterruptX(&H21, InregsX, OutregsX)

    ' ambiguate filename.
    CALL GetShortFilename(Filename2$)
    ASCIIZ = ASCIIZ3
 END IF

 ' open filename for append.
 OPEN Filename2$ FOR APPEND AS #1
 IF ErrorTrap THEN
    CALL MultiFileFunction(2)
    GOSUB RedrawScreen0
    ValidFile = False
 END IF
 RETURN

' print screen of current page of hex/ascii values to printer.
PrintScreen:
 ' get printer port.
 GOSUB OpenPrinter
 IF ValidPrint = False THEN
    RETURN
 END IF

 ' start print.
 CALL PrintSub(1)
 CALL displaypagebyte
 StatusMessage = "Screen dumped to printer port:" + STR$(PortNumber) + "."
 CALL DisplayStatus1
 RETURN

' print file of current hex/ascii values to printer.
PrintFile:
 ' get printer port.
 GOSUB OpenPrinter
 IF ValidPrint = False THEN
    RETURN
 END IF

 ' start print.
 CALL PrintSub(2)
 CALL displaypagebyte
 IF FileDumped THEN
    StatusMessage = "File dumped to printer port:" + STR$(PortNumber) + "."
 ELSE
    StatusMessage = "Partial file dumped to printer port:" + STR$(PortNumber) + "."
 END IF
 CALL DisplayStatus1
 RETURN

' print help screens.
PrintHelp:
 ' get printer port.
 GOSUB OpenPrinter
 IF ValidPrint = False THEN
    RETURN
 END IF

 ' start print.
 CALL PrintSub(3)
 CALL displayfilename
 RETURN

' get printer port.
OpenPrinter:
 ValidPrint = False
 CALL clearstatus
 PRINTf "Enter printer port(1-3)?"
 PortNumber$ = LineInput$(2, 29, 1)
 IF PortNumber$ = Nul THEN
    PortNummber = PrinterPort
 ELSE
    PortNumber = INT(VAL(PortNumber$)+.5)
 END IF
 IF PortNumber >= 1 AND PortNumber <= 3 THEN
    Check.Disk = True
    Disk.Ready = False
    CLOSE #1
    Filename2$ = "LPT" + MID$(STR$(PortNumber), 2) + ":"
    OPEN Filename2$ FOR OUTPUT AS #1
    Check.Disk = False
    IF Disk.Ready THEN
       StatusMessage = "Invalid printer port number."
       CALL DisplayStatus1
       RETURN
    END IF
 ELSE
    StatusMessage = "Invalid printer port number."
    CALL DisplayStatus1
    RETURN
 END IF
 ValidPrint = True
 RETURN

' add multiple specified byte to file and locate there.
AppendAnyByte:
 CALL clearstatus
 PRINTf "Enter number of bytes?"
 ByteString$ = LineInput$(2, 27, 10)
 IF LEN(ByteString$) = False THEN
    StatusMessage = "Invalid append bytes."
    CALL displaystatus2
    RETURN
 END IF
 IF UCASE$(LEFT$(ByteString$, 1)) = "H" THEN
    CALL CheckHexValue(ByteString$)
    IF ValidHexValue = False THEN
       RETURN
    END IF
    NumBytes2# = HexValue - 1#
 ELSE
    CALL CheckASCIIValue(ByteString$)
    IF ValidASCIIValue = False THEN
       RETURN
    END IF
    NumBytes2# = ASCIIValue3
 END IF
 CALL clearstatus
 ValidByteString = False
 PRINTf "Enter hex byte?"
 ByteString$ = LineInput$(2, 20, 2)
 IF LEN(ByteString$) = 2 THEN
    AllowWildcard = False
    CALL CheckHexBytes(ByteString$)
    IF ValidByteString THEN
       Byte2 = VAL("&H" + ByteString$)
    END IF
 END IF
 IF ValidByteString = False THEN
    StatusMessage = "Invalid append hex value."
    CALL displaystatus2
    RETURN
 END IF
 CALL DisplayX
 COLORf White
 PRINTf " Appending specified bytes to file:"
 CursorLocation = 53 + LEN(RTRIM$(Filename))
 GOTO AppendBytes2

' add multiple null bytes to file and locate there.
AppendNullBytes:
 CALL clearstatus
 PRINTf "Enter number of bytes?"
 ByteString$ = LineInput$(2, 27, 10)
 IF LEN(ByteString$) = False THEN
    StatusMessage = "Invalid append bytes."
    CALL displaystatus2
    RETURN
 END IF
 IF UCASE$(LEFT$(ByteString$, 1)) = "H" THEN
    CALL CheckHexValue(ByteString$)
    IF ValidHexValue = False THEN
       RETURN
    END IF
    NumBytes2# = HexValue - 1#
 ELSE
    CALL CheckASCIIValue(ByteString$)
    IF ValidASCIIValue = False THEN
       RETURN
    END IF
    NumBytes2# = ASCIIValue3
 END IF
 Byte2 = False
 CALL DisplayX
 COLORf White
 PRINTf " Appending null bytes to file:"
 CursorLocation = 48 + LEN(RTRIM$(Filename))

' append bytes past end of file.
AppendBytes2:
 LastPage = FilePage
 CALL ClearPageByte
 PercentDisplayed! = SFalse
 LOCATEf 2, CursorLocation, 0
 PRINTf " 0%"
 BlockBytes# = INT(NumBytes2# / BlockByteLength)
 RemainingBytes# = NumBytes2# MOD BlockByteLength
 ' write byte blocks.
 IF BlockBytes# > 0# THEN
    BlockByte = STRING$(BlockByteLength, CHR$(Byte2))
    FOR Byte1# = 1# TO BlockBytes#
       IF INKEY$ = CHR$(27) THEN
          GOTO EndAppend
       END IF
       IF INKEY$ = CHR$(0) + CHR$(0) THEN
          GOTO EndAppend
       END IF
       IF FileLength + BlockByteLength <= 2147483647# THEN
          SeekPosition = FileLength + 1#
          FileLength = FileLength + BlockByteLength
          CALL lseekfile
          GOSUB WriteBlockFile
          IF Write.Error THEN
             CALL GetFileLength(Var)
             SeekPosition = FileLength
             CALL lseekfile
             GOTO EndAppend
          END IF
          FilePosition = FileLength
          CALL CalculatePosition1
          PercentCopied! = INT(CSNG(Byte1# / BlockBytes#) * 100!)
          IF PercentCopied! > PercentDisplayed! THEN
             LOCATEf 2, CursorLocation, 0
             PRINTf STR$(PercentCopied!) + "%"
             PercentDisplayed! = PercentCopied!
          END IF
       END IF
    NEXT
 END IF
 ' write remaining bytes.
 PercentDisplayed! = SFalse
 LOCATEf 2, CursorLocation, 0
 PRINTf " 0%  "
 IF RemainingBytes# > 0# THEN
    FileByte = CHR$(Byte2)
    FOR Byte1# = 1# TO RemainingBytes#
       IF FileLength + 1# <= 2147483647# THEN
          FileLength = FileLength + 1#
          SeekPosition = FileLength
          CALL lseekfile
          CALL WriteFile
          IF Write.Error THEN
             CALL GetFileLength(Var)
             SeekPosition = FileLength
             CALL lseekfile
             GOTO EndAppend
          END IF
          FilePosition = FileLength
          CALL CalculatePosition1
          IF LastPage = FilePage THEN
             CALL ClearPageByte
          END IF
          PercentCopied! = INT(CSNG(Byte1# / RemainingBytes#) * 100!)
          IF PercentCopied! > PercentDisplayed! THEN
             LOCATEf 2, CursorLocation, 0
             PRINTf STR$(PercentCopied!) + "%"
             PercentDisplayed! = PercentCopied!
          END IF
       ELSE
          EXIT FOR
       END IF
    NEXT
 END IF
EndAppend:
 CALL CalculatePosition1
 IF LastPage <> FilePage THEN
    LastPage = FilePage
    CALL DisplayHexPage
 END IF
 CALL displaypagebyte
 CALL displayfilename
 RETURN

' add multiple bytes of an ASCII string to file and locate there.
AppendASCII:
 CALL clearstatus
 PRINTf "Enter ASCII string?"
 ByteString$ = LineInput$(2, 24, 53)
 IF LEN(ByteString$) = False THEN
    StatusMessage = "Invalid append string."
    CALL displaystatus2
    RETURN
 END IF
 NumBytes = LEN(ByteString$)
 CALL AsciiToHex2(ByteString$)
 GOTO StartAppend

' add multiple bytes to file and locate there.
'   left window specifies space-separated hex byte pairs,
'   right window specifies space-separated 3-byte ascii pairs.
AppendByte:
 CALL clearstatus
 IF CurrentWindow = False THEN
    PRINTf "Enter hex byte(s)?"
    ByteString$ = LineInput$(2, 23, 55)
    IF LEFT$(ByteString$, 1) = "+" THEN
       ByteString$ = MID$(ByteString$, 2)
       CALL CheckAsciiBytes(ByteString$)
    ELSE
       AllowWildcard = False
       CALL CheckHexBytes(ByteString$)
    END IF
 ELSE
    PRINTf "Enter ascii byte(s)?"
    ByteString$ = LineInput$(2, 25, 53)
    IF UCASE$(LEFT$(ByteString$, 1)) = "H" THEN
       ByteString$ = MID$(ByteString$, 2)
       AllowWildcard = False
       CALL CheckHexBytes(ByteString$)
    ELSE
       CALL CheckAsciiBytes(ByteString$)
    END IF
 END IF
 IF ValidByteString = False THEN
    StatusMessage = "Invalid append string."
    CALL displaystatus2
    RETURN
 END IF

' append the byte string.
StartAppend:
 LastPage = FilePage
 FOR Byte1 = 1 TO NumBytes
    CALL ClearPageByte
    Byte$ = MID$(ByteString$, (Byte1 - 1) * 2 + 1, 2)
    Byte2 = VAL("&H" + Byte$)
    GOSUB AppendIt
    IF ValidAppend = False THEN
       EXIT FOR
    END IF
 NEXT
 CALL displayfilename
 RETURN

' appends a byte to file.
AppendIt:
 ValidAppend = False
 FileByte = CHR$(Byte2)
 IF FileLength + 1# <= 2147483647# THEN
    FileLength = FileLength + 1#
    SeekPosition = FileLength
    CALL lseekfile
    CALL WriteFile
    IF Write.Error THEN
       CALL GetFileLength(Var)
       SeekPosition = FileLength
       CALL lseekfile
       CALL displaypagebyte
       RETURN
    END IF
    ValidAppend = True
    FilePosition = FileLength
    CALL CalculatePosition1
    IF LastPage <> FilePage THEN
       LastPage = FilePage
       CALL DisplayHexPage
    END IF
    CALL displaypagebyte
 END IF
 RETURN

' jump to page position in file.
JumpPage:
 CALL clearstatus
 PRINTf "Enter page position?"
 ByteString$ = LineInput$(2, 25, 10)
 IF UCASE$(LEFT$(ByteString$, 1)) = "H" THEN
    CALL CheckHexValue(ByteString$)
    IF ValidHexValue = False THEN
       RETURN
    END IF
    NextPage = HexValue
 ELSE
    CALL CheckASCIIValue(ByteString$)
    IF ValidASCIIValue = False THEN
       RETURN
    END IF
    NextPage = ASCIIValue3
 END IF
 ' check the new page is there to jump to.
 TempPosition3 = (NextPage - 1) * 320 + 1
 IF TempPosition3 >= 1 AND TempPosition3 <= FileLength THEN
    CALL ClearPageByte
    LastPage = FilePage
    ' preserve current row, column.
    FilePosition = TempPosition3
    TempPosition2 = FilePosition + (PageRow - 1) * 20 + PageColumn - 1
    IF TempPosition2 <= FileLength THEN
       FilePosition = TempPosition2
    ELSE
       FilePosition = FileLength
    END IF
    CALL CalculatePosition1
    IF LastPage <> FilePage THEN
       CALL DisplayHexPage
    END IF
    CALL displaypagebyte
    CALL displayfilename
    RETURN
 END IF
 StatusMessage = "Invalid page."
 CALL displaystatus2
 RETURN

' jump to a byte position in file.
JumpByte:
 CALL clearstatus
 PRINTf "Enter file position?"
 ByteString$ = LineInput$(2, 25, 10)
 IF UCASE$(LEFT$(ByteString$, 1)) = "H" THEN
    CALL CheckHexValue(ByteString$)
    IF ValidHexValue = False THEN
       RETURN
    END IF
    NewByte = HexValue
 ELSE
    CALL CheckASCIIValue(ByteString$)
    IF ValidASCIIValue = False THEN
       RETURN
    END IF
    NewByte = ASCIIValue3
 END IF
 IF NewByte >= 1 AND NewByte <= FileLength THEN
    IF NewByte <> FilePosition THEN
       CALL ClearPageByte
       LastPage = FilePage
       FilePosition = NewByte
       CALL CalculatePosition1
       IF LastPage <> FilePage THEN
          CALL DisplayHexPage
       END IF
       CALL displaypagebyte
    END IF
    CALL displayfilename
    RETURN
 END IF
 StatusMessage = "Invalid byte."
 CALL displaystatus2
 RETURN

' undo a byte from stored arrays of previous position/byte value.
UndoByte:
 IF CurrentUndo = 0! THEN
    StatusMessage = "No bytes to restore."
    CALL displaystatus2
    RETURN
 END IF
 IF CurrentUndo > 0! THEN
    NextUndo! = CurrentUndo
    GOSUB UndoIt
 END IF
 RETURN

' undo all bytes from stored arrays of previous position/byte value.
UndoAll:
 IF CurrentUndo = 0! THEN
    StatusMessage = "No bytes to restore."
    CALL displaystatus2
    RETURN
 END IF
 IF CurrentUndo > 0! THEN
    TotalUndos! = CurrentUndo
    FOR NextUndo! = TotalUndos! TO 1! STEP -1
       GOSUB UndoIt
    NEXT
 END IF
 RETURN

' Undo one byte.
UndoIt:
 CALL ClearPageByte
 LastPage = FilePage
 FileByte = CHR$(UndoByte%(NextUndo!))
 FilePosition = UndoPosition#(NextUndo!)
 CurrentUndo = CurrentUndo - 1!
 SeekPosition = FilePosition
 CALL lseekfile
 CALL WriteFile
 CALL CalculatePosition1
 IF LastPage <> FilePage THEN
    LastPage = FilePage
    CALL DisplayHexPage
 END IF
 CALL displaypagebyte
 RETURN

REM Cursor key routines:

' move cursor up.
CursorUp:
 IF PageRow - 1 >= 1 THEN
    CALL ClearPageByte
    PageRow = PageRow - 1
    FilePosition = FilePosition - 20
    CALL displaypagebyte
    RETURN
 END IF
 CALL ClearPageByte
 CALL displaypagebyte
 RETURN

' move cursor down.
CursorDown:
 IF PageRow + 1 <= 16 THEN
    IF FilePosition + 20 <= FileLength THEN
       CALL ClearPageByte
       PageRow = PageRow + 1
       FilePosition = FilePosition + 20
       IF FilePosition > FileLength THEN
          PageColumn = PageColumn - 1
          FilePosition = FilePosition - 1
       END IF
       CALL displaypagebyte
       RETURN
    END IF
 END IF
 CALL ClearPageByte
 CALL displaypagebyte
 RETURN

' move cursor left.
CursorLeft:
 IF PageColumn - 1 >= 1 THEN
    IF CopyPositionStart > False THEN
       IF PageColumn = 20 THEN
          CALL ClearPageByte
          CALL displaypagebyte
          RETURN
       END IF
       IF FilePosition = FileLength THEN
          CALL ClearPageByte
          CALL displaypagebyte
          RETURN
       END IF
    END IF
    CALL ClearPageByte
    PageColumn = PageColumn - 1
    FilePosition = FilePosition - 1
    CALL displaypagebyte
 ELSE
    CALL ClearPageByte
    IF PageColumn > 1 THEN
       PageColumn = 1
       FilePosition = FilePosition - 1
    END IF
    CALL displaypagebyte
 END IF
 RETURN

' move cursor right.
CursorRight:
 IF PageColumn + 1 <= 20 THEN
    IF FilePosition + 1 <= FileLength THEN
       CALL ClearPageByte
       IF PageColumn < 20 THEN
          IF FilePosition + 1 <= FileLength THEN
             PageColumn = PageColumn + 1
             FilePosition = FilePosition + 1
          END IF
       END IF
       CALL displaypagebyte
       RETURN
    END IF
 END IF
 CALL ClearPageByte
 CALL displaypagebyte
 RETURN

' end key.
EndKey:
 ' set to column 20
 IF PageColumn < 20 THEN
    ' check column 20 is positioned before end of file
    IF FilePosition - PageColumn + 20 <= FileLength THEN
       CALL ClearPageByte
       FilePosition = FilePosition - PageColumn + 20
       PageColumn = 20
       CALL displaypagebyte
    ELSE
       ' position end of file
       CALL ClearPageByte
       FilePosition = FileLength
       CALL CalculatePosition1
       CALL displaypagebyte
    END IF
 ELSE
    CALL ClearPageByte
    CALL displaypagebyte
    CALL locatecursor
 END IF
 RETURN

' home key.
HomeKey:
 IF PageColumn > 1 THEN
    CALL ClearPageByte
    FilePosition = FilePosition - PageColumn + 1
    PageColumn = 1
    CALL displaypagebyte
 ELSE
    CALL ClearPageByte
    IF PageColumn > 1 THEN
       PageColumn = 1
       FilePosition = FilePosition - 1
    END IF
    CALL displaypagebyte
    CALL locatecursor
 END IF
 RETURN

' control-left key.
CtrlLeftKey:
 NewColumn = False
 SELECT CASE PageColumn
 CASE 2, 3, 4
    NewColumn = 1
 CASE 5
    NewColumn = 4
 CASE 6, 7, 8
    NewColumn = 5
 CASE 9
    NewColumn = 8
 CASE 10, 11, 12
    NewColumn = 9
 CASE 13
    NewColumn = 12
 CASE 14, 15, 16
    NewColumn = 13
 CASE 17
    NewColumn = 16
 CASE 18, 19, 20
    NewColumn = 17
 END SELECT
 IF NewColumn > False THEN
    CALL ClearPageByte
    FilePosition = FilePosition - PageColumn + NewColumn
    PageColumn = NewColumn
    CALL displaypagebyte
 END IF
 RETURN

' control-right key.
CtrlRightKey:
 NewColumn = False
 SELECT CASE PageColumn
 CASE 1, 2, 3
    NewColumn = 4
 CASE 4
    NewColumn = 5
 CASE 5, 6, 7
    NewColumn = 8
 CASE 8
    NewColumn = 9
 CASE 9, 10, 11
    NewColumn = 12
 CASE 12
    NewColumn = 13
 CASE 13, 14, 15
    NewColumn = 16
 CASE 16
    NewColumn = 17
 CASE 17, 18, 19
    NewColumn = 20
 END SELECT
 IF NewColumn THEN
    ' check new column is positioned before end of file
    IF FilePosition - PageColumn + NewColumn <= FileLength THEN
       CALL ClearPageByte
       FilePosition = FilePosition - PageColumn + NewColumn
       PageColumn = NewColumn
       CALL displaypagebyte
    ELSE
       ' position end of file
       CALL ClearPageByte
       FilePosition = FileLength
       CALL CalculatePosition1
       CALL displaypagebyte
    END IF
 END IF
 RETURN

' control-end key.
CtrlEndKey:
 IF FilePosition <> FileLength THEN
    LastPage = FilePage
    CALL ClearPageByte
    FilePosition = FileLength
    CALL CalculatePosition1
    IF LastPage <> FilePage THEN
       CALL DisplayHexPage
    END IF
    CALL displaypagebyte
 ELSE
    CALL ClearPageByte
    CALL displaypagebyte
    CALL locatecursor
 END IF
 RETURN

' control-home key.
CtrlHomeKey:
 IF FilePosition <> 1 THEN
    LastPage = FilePage
    CALL ClearPageByte
    FilePosition = 1
    CALL CalculatePosition1
    IF LastPage <> FilePage THEN
       CALL DisplayHexPage
    END IF
    CALL displaypagebyte
 END IF
 RETURN

' move one page up.
PageUp:
 ' check there is a page to move up.
 IF FilePosition - 320 >= 1 THEN
    FilePage = FilePage - 1
    FilePosition = FilePosition - 320
    CALL DisplayHexPage
    CALL displaypagebyte
 ELSE
    ' check there is a first position on the first page to move up to.
    IF FilePosition > 1 THEN
       CALL ClearPageByte
       PageRow = 1
       PageColumn = 1
       FilePosition = 1
       CALL displaypagebyte
    ELSE
       CALL ClearPageByte
       CALL displaypagebyte
       CALL locatecursor
    END IF
 END IF
 RETURN

' move one page down.
PageDown:
 ' check there is a page to move down.
 IF FilePosition + 320 <= FileLength THEN
    FilePage = FilePage + 1
    FilePosition = FilePosition + 320
    CALL DisplayHexPage
    CALL displaypagebyte
 ELSE
    ' check there is a last position on the last page to move down to.
    IF FilePosition < FileLength THEN
       CALL ClearPageByte
       FilePosition = FileLength
       CALL CalculatePosition1
       CALL DisplayHexPage
       CALL displaypagebyte
    ELSE
       CALL ClearPageByte
       CALL displaypagebyte
       CALL locatecursor
    END IF
 END IF
 RETURN

REM More routines:

' switch editing windows.
TabWindow:
 IF CurrentWindow2 = False THEN
    CurrentWindow = NOT CurrentWindow
    CALL DisplayPosition
    CALL locatecursor2
 END IF
 RETURN

' switch right editing window.
ToggleRight:
 CurrentWindow = False
 CurrentWindow2 = NOT CurrentWindow2
 CALL RedrawRightWindow
 CALL displaypagebyte
 RETURN

' change the byte value at the current position.
'   change by hex value in left window,
'   by ascii value in right window.
'   overrides with +255 in left window,
'   and with hff in right window.
ChangeHexValue:
 CALL clearstatus
 FileByte = Nul
 ValidByteString = False
 IF CurrentWindow = False THEN
    PRINTf "Enter hex byte value?"
    ByteString$ = LineInput$(2, 26, 4)
    IF LEFT$(ByteString$, 1) = "+" THEN
       ByteString$ = MID$(ByteString$, 2)
       IF LEN(ByteString$) = 3 THEN
          NewValue = INT(VAL(ByteString$)+.5)
          IF NewValue >= False AND NewValue <= 255 THEN
             ValidByteString = True
             FileByte = CHR$(NewValue)
          END IF
       END IF
    ELSE
       IF LEN(ByteString$) = 2 THEN
          AllowWildcard = False
          CALL CheckHexBytes(ByteString$)
          IF ValidByteString THEN
             NewValue = VAL("&H" + ByteString$)
             FileByte = CHR$(NewValue)
          END IF
       END IF
    END IF
 ELSE
    PRINTf "Enter ascii byte value?"
    ByteString$ = LineInput$(2, 28, 4)
    IF UCASE$(LEFT$(ByteString$, 1)) = "H" THEN
       ByteString$ = MID$(ByteString$, 2)
       IF LEN(ByteString$) = 2 THEN
          AllowWildcard = False
          CALL CheckHexBytes(ByteString$)
          IF ValidByteString THEN
             NewValue = VAL("&H" + ByteString$)
             FileByte = CHR$(NewValue)
          END IF
       END IF
    ELSE
       IF LEN(ByteString$) = 3 THEN
          NewValue = INT(VAL(ByteString$)+.5)
          IF NewValue >= False AND NewValue <= 255 THEN
             ValidByteString = True
             FileByte = CHR$(NewValue)
          END IF
       END IF
    END IF
 END IF
 IF ValidByteString = False THEN
    StatusMessage = "Invalid byte."
    CALL displaystatus2
    RETURN
 END IF
 GOSUB StoreUndo
 SeekPosition = FilePosition
 CALL lseekfile
 CALL WriteFile
 CALL displaypagebyte
 CALL displayfilename
 RETURN

' change the ascii values with string starting at the current position.
ChangeASCIIValues:
 CALL clearstatus
 PRINTf "Enter ascii value(s)?"
 ByteString$ = LineInput$(2, 26, 52)
 IF LEN(ByteString$) = False THEN
    StatusMessage = "Invalid ascii string."
    CALL displaystatus2
    RETURN
 END IF
 ' init file.
 IF FileLength = 0 THEN
    Byte2 = 0
    GOSUB AppendIt
    IF ValidAppend = False THEN
       RETURN
    END IF
 ELSE
    ' store all current values.
    StorePosition = FilePosition
    FOR NumberBytes = 1 TO LEN(ByteString$)
       SeekPosition = FilePosition
       CALL lseekfile
       CALL readfile
       FileByte = Buffer
       AsciiValue = ASC(FileByte)
       GOSUB StoreUndo
       IF FilePosition = FileLength THEN
          EXIT FOR
       END IF
       FilePosition = FilePosition + 1
    NEXT
    FilePosition = StorePosition
 END IF
 ' write ascii string.
 ValidAppend = True
 FOR NumberBytes = 1 TO LEN(ByteString$)
    IF FilePosition > FileLength THEN
       Byte2 = 0
       GOSUB AppendIt
       IF ValidAppend = False THEN
          EXIT FOR
       END IF
    END IF
    FileByte = MID$(ByteString$, NumberBytes, 1)
    SeekPosition = FilePosition
    CALL lseekfile
    CALL WriteFile
    CALL CalculatePosition1
    IF LastPage <> FilePage THEN
       LastPage = FilePage
       CALL DisplayHexPage
    END IF
    CALL ClearPageByte
    FilePosition = FilePosition + 1
 NEXT
 IF ValidAppend THEN
    FilePosition = FilePosition - 1
 END IF
 CALL displaypagebyte
 CALL displayfilename
 RETURN

' change the hex values with string starting at the current position.
ChangeHEXValues:
 CALL clearstatus
 PRINTf "Enter HEX value(s)?"
 ByteString$ = LineInput$(2, 24, 54)
 IF LEN(ByteString$) = False THEN
    StatusMessage = "Invalid hex string."
    CALL displaystatus2
    RETURN
 END IF
 ' check hex string.
 AllowWildcard = False
 CALL CheckHexBytes(ByteString$)
 IF ValidByteString = False THEN
    StatusMessage = "Invalid hex string."
    CALL displaystatus2
    RETURN
 END IF
 ' init file.
 IF FileLength = 0 THEN
    Byte2 = 0
    GOSUB AppendIt
    IF ValidAppend = False THEN
       RETURN
    END IF
 ELSE
    ' store all current values.
    StorePosition = FilePosition
    FOR NumberBytes = 1 TO LEN(ByteString$) STEP 2
       SeekPosition = FilePosition
       CALL lseekfile
       CALL readfile
       FileByte = Buffer
       AsciiValue = ASC(FileByte)
       GOSUB StoreUndo
       IF FilePosition = FileLength THEN
          EXIT FOR
       END IF
       FilePosition = FilePosition + 1
    NEXT
    FilePosition = StorePosition
 END IF
 ' write hex string.
 ValidAppend = True
 FOR NumberBytes = 1 TO LEN(ByteString$) STEP 2
    IF FilePosition > FileLength THEN
       Byte2 = 0
       GOSUB AppendIt
       IF ValidAppend = False THEN
          EXIT FOR
       END IF
    END IF
    FileByte = CHR$(VAL("&H" + MID$(ByteString$, NumberBytes, 2)))
    SeekPosition = FilePosition
    CALL lseekfile
    CALL WriteFile
    CALL CalculatePosition1
    IF LastPage <> FilePage THEN
       LastPage = FilePage
       CALL DisplayHexPage
    END IF
    CALL ClearPageByte
    FilePosition = FilePosition + 1
 NEXT
 IF ValidAppend THEN
    FilePosition = FilePosition - 1
 END IF
 CALL displaypagebyte
 CALL displayfilename
 RETURN

' record the current byte position and value,
'   increment the number of undos in record structure array.
StoreUndo:
 IF CurrentUndo < 1048576! THEN
    CurrentUndo = CurrentUndo + 1!
    GET #6, CurrentUndo, UndoFile
    UndoFile.UndoByte1(CurrentFile) = AsciiValue
    UndoFile.UndoPosition1(CurrentFile) = FilePosition
    PUT #6, CurrentUndo, UndoFile
 END IF
 RETURN

REM Drop down input menu.

Menu2:
 CALL SMouse
 CALL DropDownMenu
 GOSUB RestorePosition
 IF CurrentMenu = False THEN
    CALL locatecursor
    GOTO StartLoop
 END IF
 ' reset mouse activity.
 CALL HMouse
 SELECT CASE CurrentMenu
 CASE 1
    SELECT CASE CurrentMenuSelection
    CASE 1 ' Alt-N  Open new file
       IF NumberFiles < MaxFiles THEN
          CALL MultiFileFunction(2)
          GOSUB NewFile
          GOTO BeginFile
       END IF
    CASE 2 ' Alt-W  Close file
       GOSUB CloseCurrentFile
       IF NumberFiles = False THEN
          GOSUB NewFile
          GOTO BeginFile
       END IF
    CASE 3 ' Alt-0  Close All Files
       GOSUB CloseAllFiles
       GOSUB ResetAllAttributes
       GOSUB NewFile
       NumberFiles = False
       GOTO BeginFile
    CASE 4 ' Alt-V  View files
       CALL ViewFiles(Var)
       IF Var > False THEN
          NextFile = Var
          GOSUB SearchFile3
       END IF
       CALL displayfilename
    CASE 5 ' Alt-Y  Toggle window
       IF FileLength THEN
          GOSUB ToggleRight
       END IF
    CASE 6 ' Alt-X  Exit program
       GOTO StopLabel
    END SELECT
 CASE 2
    SELECT CASE CurrentMenuSelection
    CASE 1 ' Alt-C  ANSI Chart
       GOSUB ANSIChart
    CASE 2 ' Alt-H  HEX Chart
       GOSUB HEXChart
    END SELECT
 CASE 3
    IF FileLength THEN
       SELECT CASE CurrentMenuSelection
       CASE 1 ' Alt-D  HEX Screen Dump
          GOSUB DumpScreen
       CASE 2 ' Alt-E  HEX File Dump
          GOSUB DumpHEXFile
       END SELECT
    END IF
 CASE 4
    SELECT CASE CurrentMenuSelection
    CASE 1 ' Alt-A  Append bytes
       GOSUB AppendByte
    CASE 2 ' Alt-M  Append ASCII string
       GOSUB AppendASCII
    CASE 3 ' Alt-R  Redraw screen
       CALL MultiFileFunction(2)
       GOSUB RedrawScreen0
    CASE 4 ' Alt-U  Undo last byte
       IF FileLength THEN
          GOSUB UndoByte
       END IF
    CASE 5 ' Alt-Z  Undo All bytes
       IF FileLength THEN
          GOSUB UndoAll
       END IF
    END SELECT
 CASE 5
    IF FileLength THEN
       SELECT CASE CurrentMenuSelection
       CASE 1 ' Alt-J  Jump to byte
          GOSUB JumpByte
       CASE 2 ' Alt-L  Jump to page
          GOSUB JumpPage
       END SELECT
    END IF
 CASE 6
    IF FileLength THEN
       SELECT CASE CurrentMenuSelection
       CASE 1 ' Alt-P  HEX Screen Print
          GOSUB PrintScreen
       CASE 2 ' Alt-T  HEX File Print
          GOSUB PrintFile
       END SELECT
    END IF
 CASE 7
    IF FileLength THEN
       SELECT CASE CurrentMenuSelection
       CASE 1 ' Alt-K  Search for string
          GOSUB SearchASCII
       CASE 2 ' Alt-S  Search for bytes
          GOSUB SearchBytes
       END SELECT
    END IF
 END SELECT
 ' reset mouse activity.
 CALL SMouse
 GOSUB RestorePosition
 CALL locatecursor
 GOTO StartLoop

REM Long filename routines:

REM open file based on Windows or DOS loaded.
REM   read file attribute,
REM   reset and store read-only bit,
REM   create file if nonexistent.
REM
REM returns: ValidFile=True if open successful.
REM          FileLocked=True if file locked.

OpenFile:
 IF Windows.Detected THEN
    GOSUB OpenFile1
 ELSE
    GOSUB OpenFile2
 END IF
 RETURN

REM open Windows file.
OpenFile1:
 ErrorTrap = False
 FileLocked = False
 ValidFile = False
 FileAttribute = False
 GOSUB GetWINAttribute

 ' can't read attribute of devices.
 IF (OutregsX.Flags AND &H1) = &H1 THEN
    ' check file nonexistent.
    IF (OutregsX.AX) <> 2 THEN
       RETURN
    END IF
 END IF
 FileAttribute = OutregsX.CX

 ' reset read-only bit before opening file for read-write.
 IF (FileAttribute AND 1) = 1 THEN
    GOSUB SetWINAttribute
 END IF
 Check.Disk = True
 Disk.Ready = False
 GOSUB OpenWINFile
 Check.Disk = False
 IF (OutregsX.Flags AND &H1) = &H0 THEN
    IF ErrorTrap = 70 THEN
       GOSUB CloseIt
       Check.Disk = True
       Disk.Ready = False
       GOSUB OpenWINFile2
       Check.Disk = False
       FileLocked = True
    END IF
    ValidFile = True
    RETURN
 END IF

 ' check file nonexistent.
 IF (OutregsX.AX) <> 2 THEN
    RETURN
 END IF
 GOSUB GetFilename
 GOSUB CreateWinFile
 IF (OutregsX.Flags AND &H1) = &H1 THEN
    RETURN
 END IF
 GOSUB CloseIt
 IF (OutregsX.Flags AND &H1) = &H1 THEN
    RETURN
 END IF
 GOSUB OpenWINFile
 IF (OutregsX.Flags AND &H1) = &H1 THEN
    RETURN
 END IF
 IF ErrorTrap = 70 THEN
    FileLocked = True
 END IF
 FileAttribute = False
 ValidFile = True
 RETURN

REM open DOS file.
OpenFile2:
 ErrorTrap = False
 FileLocked = False
 ValidFile = False
 FileAttribute = False
 GOSUB GetDOSAttribute

 ' can't read attribute of devices.
 IF (OutregsX.Flags AND &H1) = &H1 THEN
    ' check file nonexistent.
    IF (OutregsX.AX) <> 2 THEN
       RETURN
    END IF
 END IF
 FileAttribute = OutregsX.CX

 ' reset read-only bit before opening file for read-write.
 IF (FileAttribute AND 1) = 1 THEN
    GOSUB SetDOSAttribute
 END IF
 Check.Disk = True
 Disk.Ready = False
 GOSUB OpenDOSFile
 Check.Disk = False
 IF (OutregsX.Flags AND &H1) = &H0 THEN
    IF ErrorTrap = 70 THEN
       GOSUB CloseIt
       Check.Disk = True
       Disk.Ready = False
       GOSUB OpenDOSFile2
       Check.Disk = False
       FileLocked = True
    END IF
    ValidFile = True
    RETURN
 END IF

 ' check file nonexistent.
 IF (OutregsX.AX) <> 2 THEN
    RETURN
 END IF
 GOSUB GetFilename
 GOSUB CreateDOSFile
 IF (OutregsX.Flags AND &H1) = &H1 THEN
    RETURN
 END IF
 GOSUB CloseIt
 IF (OutregsX.Flags AND &H1) = &H1 THEN
    RETURN
 END IF
 GOSUB OpenDOSFile
 IF (OutregsX.Flags AND &H1) = &H1 THEN
    RETURN
 END IF
 IF ErrorTrap = 70 THEN
    FileLocked = True
 END IF
 FileAttribute = False
 ValidFile = True
 RETURN

REM check command line and create filename.
GetFilename:
 IF LEFT$(Filename, 2) <> "\\" THEN
    IF LEFT$(Filename, 1) = "\" THEN
       ASCIIZ = LEFT$(CURDIR$, 2) + RTRIM$(Filename) + CHR$(0)
    ELSE
       IF MID$(Filename, 2, 1) <> ":" THEN
          IF RIGHT$(CURDIR$, 1) = "\" THEN
             ASCIIZ = CURDIR$ + RTRIM$(Filename) + CHR$(0)
          ELSE
             ASCIIZ = CURDIR$ + "\" + RTRIM$(Filename) + CHR$(0)
          END IF
       END IF
    END IF
 END IF
 RETURN

REM open file read/write in Windows.
OpenWINFile:
 InregsX.AX = &H716C
 InregsX.BX = &H2
 InregsX.CX = FileAttribute
 InregsX.DX = &H1
 InregsX.DS = VARSEG(ASCIIZ)
 InregsX.SI = VARPTR(ASCIIZ)
 InregsX.DI = &H1
 CALL InterruptX(&H21, InregsX, OutregsX)
 Handle = OutregsX.AX
 RETURN

REM open file read-only in Windows.
OpenWINFile2:
 InregsX.AX = &H716C
 InregsX.BX = &H44
 InregsX.CX = FileAttribute
 InregsX.DX = &H1
 InregsX.DS = VARSEG(ASCIIZ)
 InregsX.SI = VARPTR(ASCIIZ)
 InregsX.DI = &H1
 CALL InterruptX(&H21, InregsX, OutregsX)
 Handle = OutregsX.AX
 RETURN

REM get file attribute.
GetWINAttribute:
 InregsX.AX = &H7143
 InregsX.BX = &H0
 InregsX.DS = VARSEG(ASCIIZ)
 InregsX.DX = VARPTR(ASCIIZ)
 CALL InterruptX(&H21, InregsX, OutregsX)
 RETURN

REM reset read-only bit.
SetWINAttribute:
 Attribute = (FileAttribute AND NOT 1)
 InregsX.AX = &H7143
 InregsX.BX = &H1
 InregsX.CX = Attribute
 InregsX.DS = VARSEG(ASCIIZ)
 InregsX.DX = VARPTR(ASCIIZ)
 CALL InterruptX(&H21, InregsX, OutregsX)
 RETURN

REM open file read/write in DOS.
OpenDOSFile:
 InregsX.AX = &H6C00
 InregsX.BX = &H1002
 InregsX.CX = FileAttribute
 InregsX.DX = &H1
 InregsX.DS = VARSEG(ASCIIZ)
 InregsX.SI = VARPTR(ASCIIZ)
 CALL InterruptX(&H21, InregsX, OutregsX)
 Handle = OutregsX.AX
 RETURN

REM open file read-only in DOS.
OpenDOSFile2:
 InregsX.AX = &H6C00
 InregsX.BX = &H1000
 InregsX.CX = FileAttribute
 InregsX.DX = &H1
 InregsX.DS = VARSEG(ASCIIZ)
 InregsX.SI = VARPTR(ASCIIZ)
 CALL InterruptX(&H21, InregsX, OutregsX)
 Handle = OutregsX.AX
 RETURN

REM get file attribute.
GetDOSAttribute:
 InregsX.AX = &H4300
 InregsX.DS = VARSEG(ASCIIZ)
 InregsX.DX = VARPTR(ASCIIZ)
 CALL InterruptX(&H21, InregsX, OutregsX)
 RETURN

REM reset read-only bit.
SetDOSAttribute:
 Attribute = (FileAttribute AND NOT 1)
 InregsX.AX = &H4301
 InregsX.CX = Attribute
 InregsX.DS = VARSEG(ASCIIZ)
 InregsX.DX = VARPTR(ASCIIZ)
 CALL InterruptX(&H21, InregsX, OutregsX)
 RETURN

REM create new file in Windows.
CreateWinFile:
 InregsX.AX = &H716C
 InregsX.BX = &H2
 InregsX.CX = &H0
 InregsX.DX = &H10
 InregsX.DS = VARSEG(ASCIIZ)
 InregsX.SI = VARPTR(ASCIIZ)
 InregsX.DI = &H1
 CALL InterruptX(&H21, InregsX, OutregsX)
 Handle = OutregsX.AX
 RETURN

REM create new file in DOS.
CreateDOSFile:
 InregsX.AX = &H6C00
 InregsX.BX = &H1000
 InregsX.CX = &H0
 InregsX.DX = &H10
 InregsX.DS = VARSEG(ASCIIZ)
 InregsX.SI = VARPTR(ASCIIZ)
 CALL InterruptX(&H21, InregsX, OutregsX)
 Handle = OutregsX.AX
 RETURN

REM close all files.
CloseAllFiles:
 FOR Var = 1 TO NumberFiles
    Handle = File(Var).Handle
    FileLocked = File(Var).FileLocked
    FileLength = File(Var).FileLength
    IF Handle THEN
       GOSUB CloseFile
    END IF
 NEXT
 RETURN

REM check close file.
CloseFile:
 IF FileLocked = True THEN
    RETURN
 END IF

REM close dos file.
CloseIt:
 InregsX.AX = &H3E00
 InregsX.BX = Handle
 CALL InterruptX(&H21, InregsX, OutregsX)
 RETURN

REM write byte block to file.
WriteBlockFile:
 IF FileLocked THEN
    Write.Error = True
    RETURN
 END IF
 Write.Error = False
 InregsX.AX = &H4000
 InregsX.BX = Handle
 InregsX.CX = BlockByteLength
 InregsX.DS = VARSEG(BlockByte)
 InregsX.DX = VARPTR(BlockByte)
 CALL InterruptX(&H21, InregsX, OutregsX)
 IF (OutregsX.Flags AND &H1) = &H1 THEN
    Write.Error = True
 ELSE
    IF OutregsX.AX = False THEN
       Write.Error = True
    END IF
 END IF
 RETURN

REM get netpath from filename
ParseNetPath: ' \\serv1\share\filename.ext
 NetPath$ = Nul
 IF LEFT$(Filename, 2) = "\\" THEN
    Net1 = INSTR(3, Filename, "\")
    IF Net1 THEN
       Net2 = INSTR(Net1 + 1, Filename, "\")
       IF Net2 THEN
          NetPath$ = LEFT$(Filename, Net2 - 1)
       END IF
    END IF
 END IF
 RETURN

REM conanicalize filename for display purposes.
ShortFilename:
 ' read ambiguated 8.3 filename.
 ASCIIZ3 = ASCIIZ
 CALL GetShortFilename(Short.Filename$)
 ASCIIZ = ASCIIZ3

 ' store 64-byte filename.
 Filename = Short.Filename$
 ShortFilename = Filename
 CurrentNetPath = NetPath$

 ' read conanicalized 8.3 filename.
 Filename = Conanicalize$(Filename)
 RETURN

REM restore all read-only attributes.
ResetAllAttributes:
 FOR Var = 1 TO NumberFiles
    ' reset file attribute.
    ASCIIZ = File(Var).ASCIIZ
    FileAttribute = File(Var).FileAttribute
    Handle = File(Var).Handle
    IF Handle THEN
       GOSUB ResetAttribute
    END IF
 NEXT
 RETURN

REM restore read-only attribute.
ResetAttribute:
 IF (FileAttribute AND &H1) = &H1 THEN
    IF Windows.Detected THEN
       GOSUB ResetWINAttribute
    ELSE
       GOSUB ResetDOSAttribute
    END IF
 END IF
 RETURN

REM restore windows read-only attribute.
ResetWINAttribute:
 InregsX.AX = &H7143
 InregsX.BX = &H1
 InregsX.CX = FileAttribute
 InregsX.DS = VARSEG(ASCIIZ)
 InregsX.DX = VARPTR(ASCIIZ)
 CALL InterruptX(&H21, InregsX, OutregsX)
 RETURN

REM restore DOS read-only attribute.
ResetDOSAttribute:
 InregsX.AX = &H4301
 InregsX.CX = FileAttribute
 InregsX.DS = VARSEG(ASCIIZ)
 InregsX.DX = VARPTR(ASCIIZ)
 CALL InterruptX(&H21, InregsX, OutregsX)
 RETURN

REM Shell routines:

REM Run DOS shell.
DOSshell:
 CALL MultiFileFunction(2)
 COLORf Plain
 CLS
 PRINTf "Type 'Exit' to return to editor.."
 LOCATEf 2, 1, 1
 Var$ = CURDIR$
 CALL ResetTrap
 SHELL
 CHDRIVE Var$
 CHDIR Var$
 CALL SetTrap
 GOSUB RedrawScreen0
 RETURN

REM Run screen saver.
SaveShell:
 Var$ = "scrnsave.exe"
 IF DIR$(Var$) = "" THEN
    RETURN
 END IF
 CALL MultiFileFunction(2)
 COLORf Plain
 CLS
 PRINTf "Press any key to return to editor.."
 LOCATEf 2, 1, 1
 CALL ResetTrap
 SHELL Var$
 CALL SetTrap
 DO
    IF INKEY$ <> "" THEN
       EXIT DO
    END IF
    R = ReleaseTime
 LOOP
 GOSUB RedrawScreen0
 RETURN

REM Run DOS command in shell.
DOScommand:
 CALL MultiFileFunction(2)
 CALL clearstatus
 PRINTf "Enter DOS command?"
 InputString$ = LineInput$(2, 23, 55)
 IF LEN(InputString$) THEN
    ' open filename.
    CLOSE #1
    DOSfilename$ = "DOSEXIT" + MID$(STR$(Process.Number), 2) + ".BAT"
    OPEN DOSfilename$ FOR OUTPUT SHARED AS #1

    ' write DOS commands.
    PRINT #1, InputString$
    PRINT #1, "PAUSE"
    PRINT #1, "EXIT"
    CLOSE #1

    ' store current drive/path.
    Var$ = CURDIR$

    ' clear screen
    COLORf Plain
    CLS

    ' restore control-break.
    CALL ResetTrap

    ' shell to batch file.
    SHELL DOSfilename$

    ' restore drive/path.
    CHDRIVE Var$
    CHDIR Var$

    ' reset control-break.
    CALL SetTrap
 END IF
 GOSUB RedrawScreen0
 RETURN

REM Mouse routines:

' move mouse cursor for mouseover.
MoveMouse:
 ' check mouse boundaries.
 IF Mouse.Row = 1 THEN
    NewMenuSelection = False
    SELECT CASE Mouse.Column
    CASE 6 TO 9 ' File
       NewMenuSelection = 1
    CASE 16 TO 21 ' Charts
       NewMenuSelection = 2
    CASE 26 TO 29 ' Dump
       NewMenuSelection = 3
    CASE 36 TO 39 ' Edit
       NewMenuSelection = 4
    CASE 46 TO 49 ' Jump
       NewMenuSelection = 5
    CASE 56 TO 60 ' Print
       NewMenuSelection = 6
    CASE 64 TO 69 ' Search
       NewMenuSelection = 7
    END SELECT
    IF NewMenuSelection > False THEN
       IF NewMenuSelection <> CurrentMenu THEN
          IF CurrentMenu > False THEN
             ' remove hilight menu tab.
             CALL HMouse
             COLORf2 White, 0
             CALL DisplayTab(CurrentMenu)
             CALL SMouse
          END IF
       END IF
       CurrentMenu = NewMenuSelection
       CurrentMenuSelection = 1
    END IF
    ' hilight menu tab.
    IF CurrentMenu > False THEN
       CALL HMouse
       COLORf2 White, 1
       CALL DisplayTab(CurrentMenu)
       CALL SMouse
    END IF
    CALL locatecursor2
    RETURN
 END IF
 ' check mouse position in screen editing area.
 IF Mouse.Row > 3 AND Mouse.Row < 20 THEN
    IF Mouse.Column > 5 AND Mouse.Column < 50 THEN
       ' check screen character.
       VarX = SCREEN(Mouse.Row, Mouse.Column)
       ' reset position and return.
       IF VarX = 32 THEN ' at a space.
          GOSUB RestorePosition
          RETURN
       END IF
       ' get new position and reset there.
       IF VarX <> 32 THEN ' not a space.
          IF Old.Mouse.Row > False AND Old.Mouse.Column > False THEN
             Old.Value = False
             IF Old.Mouse.Row = PageRow + 3 THEN
                IF Old.Mouse.Column = Column + 5 OR Old.Mouse.Column = Column + 6 THEN
                   Old.Value = True
                END IF
             END IF
             IF Old.Value = False THEN
                GOSUB MoveOldPosition
             ELSE
                GOSUB RestoreOldPosition
             END IF
          END IF
          GOSUB MoveNewPosition
          RETURN
       END IF
    END IF
    ' check boundary edges.
    IF Mouse.Column = 5 OR Mouse.Column = 50 THEN
       GOSUB RestorePosition
       RETURN
    END IF
 END IF
 ' restore position.
 GOSUB RestorePosition
 CALL locatecursor2
 RETURN

' restores old mouse position.
MoveOldPosition:
 Store1 = Mouse.Row
 Store2 = Mouse.Column
 Mouse.Row = Old.Mouse.Row
 Mouse.Column = Old.Mouse.Column
 NewPosition = CalculatePosition3#
 Mouse.Row = Store1
 Mouse.Column = Store2
 TempPosition3 = NewPosition
 CALL CalculatePosition2
 Column2 = CalculateColumn2
 GOSUB CheckMousePosition
 IF MouseArea = False THEN
    CALL HMouse
    COLORf White
    LOCATEf PageRow2 + 3, Column2 + 5, 0
    VarX = SCREEN(PageRow2 + 3, Column2 + 5)
    PRINTf CHR$(VarX)
    LOCATEf PageRow2 + 3, Column2 + 6, 0
    VarX = SCREEN(PageRow2 + 3, Column2 + 6)
    PRINTf CHR$(VarX)
    CALL SMouse
 END IF
 COLORf2 Plain, 0
 RETURN

' restores original mouse position.
RestoreOldPosition:
 SeekPosition = FilePosition
 CALL lseekfile
 CALL readfile
 FileByte = Buffer
 ByteValue = ASC(FileByte)
 Column = CalculateColumn
 GOSUB CheckMousePosition2
 IF MouseArea = False THEN
    CALL HMouse
    COLORf Yellow
    LOCATEf PageRow + 3, Column + 5, 0
    PRINTf RIGHT$("00" + HEX$(ByteValue), 2)
    CALL SMouse
 END IF
 COLORf Plain
 RETURN

' moves to new mouse position.
MoveNewPosition:
 NewPosition = CalculatePosition3#
 FilePosition2 = NewPosition
 TempPosition3 = NewPosition
 CALL CalculatePosition2
 Column2 = CalculateColumn2
 GOSUB CheckMousePosition
 IF MouseArea = False THEN
    CALL HMouse
    COLORf2 White, 1
    LOCATEf PageRow2 + 3, Column2 + 5, 0
    VarX = SCREEN(PageRow2 + 3, Column2 + 5)
    PRINTf CHR$(VarX)
    LOCATEf PageRow2 + 3, Column2 + 6, 0
    VarY = SCREEN(PageRow2 + 3, Column2 + 6)
    PRINTf CHR$(VarY)
    CALL SMouse
 END IF
 COLORf2 Plain, 0
 IF Mouse.Row > 3 AND Mouse.Row < 20 THEN
    IF Mouse.Column > 5 AND Mouse.Column < 50 THEN
       Old.Mouse.Row = Mouse.Row
       Old.Mouse.Column = Mouse.Column
    END IF
 END IF
 AsciiValue2 = VAL("&H" + CHR$(VarX) + CHR$(VarY))

 ' store old values.
 StoreValue1# = FilePosition
 StoreValue2% = AsciiValue2
 ' reset to mouseover values.
 FilePosition = FilePosition2
 AsciiValue = AsciiValue2
 ' display mouseover position.
 CALL displayfilename
 ' restore old values.
 FilePosition = StoreValue1#
 AsciiValue = StoreValue2%
 RETURN

' restore screen position.
RestorePosition:
 IF Old.Mouse.Row > 3 AND Old.Mouse.Row < 20 THEN
    IF Old.Mouse.Column > 5 AND Old.Mouse.Column < 50 THEN
       VarX = SCREEN(Old.Mouse.Row, Old.Mouse.Column)
       IF VarX <> 32 THEN
          Store1 = Mouse.Row
          Store2 = Mouse.Column
          Mouse.Row = Old.Mouse.Row
          Mouse.Column = Old.Mouse.Column
          NewPosition = CalculatePosition3#
          Mouse.Row = Store1
          Mouse.Column = Store2
          TempPosition3 = NewPosition
          CALL CalculatePosition2
          Column2 = CalculateColumn2
          IF CopyPositionStart = False THEN
             GOTO RestoreMouseByte
          END IF
          IF CopyPositionStart > False THEN
             IF NewPosition >= CopyPositionStart THEN
                IF NewPosition <= CopyPositionEnd THEN
                   RETURN
                END IF
             END IF
          END IF
          GOTO RestoreMouseByte
       END IF
    END IF
 END IF
 RETURN

' check mouse position in hilighted area.
CheckMousePosition:
 MouseArea = False
 IF CopyPositionStart > False THEN
    IF TempPosition3 >= CopyPositionStart THEN
       IF TempPosition3 <= CopyPositionEnd THEN
          MouseArea = True
       END IF
    END IF
 END IF
 RETURN

' check mouse position in hilighted area.
CheckMousePosition2:
 MouseArea = False
 IF CopyPositionStart > False THEN
    IF FilePosition >= CopyPositionStart THEN
       IF TempPosition3 <= CopyPositionEnd THEN
          MouseArea = True
       END IF
    END IF
 END IF
 RETURN

' display hilighted mouse position byte.
RestoreMouseByte:
 CALL HMouse
 COLORf2 White, 1
 LOCATEf PageRow2 + 3, Column2 + 5, 0
 VarX = SCREEN(PageRow2 + 3, Column2 + 5)
 PRINTf CHR$(VarX)
 LOCATEf PageRow2 + 3, Column2 + 6, 0
 VarY = SCREEN(PageRow2 + 3, Column2 + 6)
 PRINTf CHR$(VarY)
 COLORf2 Plain, 0
 CALL SMouse
 RETURN

' process left mouse button click.
MouseButton1:
 ' stop program.
 IF Mouse.Row = 3 AND Mouse.Column = 77 THEN
    CALL HMouse
    GOTO StopLabel
 END IF
 ' help screen.
 IF Mouse.Row = 4 AND Mouse.Column = 77 THEN
    GOSUB DisplayHelp
    RETURN
 END IF
 ' page up.
 IF Mouse.Row = 4 AND Mouse.Column = 52 THEN
    IF FileLength THEN
       GOSUB PageUp
    END IF
    RETURN
 END IF
 ' page down.
 IF Mouse.Row = 20 AND Mouse.Column = 52 THEN
    IF FileLength THEN
       GOSUB PageDown
    END IF
    RETURN
 END IF
 ' line up.
 IF Mouse.Row = 5 AND Mouse.Column = 52 THEN
    IF FileLength THEN
       GOSUB CursorUp
    END IF
    RETURN
 END IF
 ' line down.
 IF Mouse.Row = 19 AND Mouse.Column = 52 THEN
    IF FileLength THEN
       GOSUB CursorDown
    END IF
    RETURN
 END IF
 ' select drop down menu.
 IF Mouse.Row = 1 THEN
    SELECT CASE Mouse.Column
    CASE 6 TO 9 ' File
       NewMenuSelection = 1
    CASE 16 TO 21 ' Charts
       NewMenuSelection = 2
    CASE 26 TO 29 ' Dump
       NewMenuSelection = 3
    CASE 36 TO 39 ' Edit
       NewMenuSelection = 4
    CASE 46 TO 49 ' Jump
       NewMenuSelection = 5
    CASE 56 TO 60 ' Print
       NewMenuSelection = 6
    CASE 64 TO 69 ' Search
       NewMenuSelection = 7
    CASE ELSE
       RETURN
    END SELECT
    IF NewMenuSelection <> CurrentMenu THEN
       IF CurrentMenu > False THEN
          CALL HMouse
          ' remove hilight menu tab.
          COLORf2 White, 0
          CALL DisplayTab(CurrentMenu)
          CALL SMouse
       END IF
    END IF
    CurrentMenu = NewMenuSelection
    CurrentMenuSelection = 1
    CALL HMouse
    ' hilight menu tab.
    COLORf2 White, 1
    CALL DisplayTab(CurrentMenu)
    CALL SMouse
    ' jump to file menu.
    CALL HMouse
    GOTO Menu2
 END IF
 ' move to new byte.
 IF Mouse.Row > 3 AND Mouse.Row < 20 THEN
    IF Mouse.Column > 5 AND Mouse.Column < 50 THEN
       VarX = SCREEN(Mouse.Row, Mouse.Column)
       IF VarX = 32 THEN
          GOSUB RestorePosition
          RETURN
       END IF
       IF VarX <> 32 THEN
          NewPosition = CalculatePosition3#
          CALL ClearPageByte
          FilePosition = NewPosition
          CALL CalculatePosition1
          CALL displaypagebyte
          CALL displayfilename
          RETURN
       END IF
    END IF
 END IF
 RETURN

' process right mouse button.
MouseButton2:
 ' move to new byte and edit.
 IF Mouse.Row > 3 AND Mouse.Row < 20 THEN
    IF Mouse.Column > 5 AND Mouse.Column < 50 THEN
       VarX = SCREEN(Mouse.Row, Mouse.Column)
       IF VarX = 32 THEN
          GOSUB RestorePosition
          RETURN
       END IF
       IF VarX <> 32 THEN
          NewPosition = CalculatePosition3#
          CALL ClearPageByte
          FilePosition = NewPosition
          CALL CalculatePosition1
          CALL displaypagebyte
          CALL displayfilename
          GOSUB ChangeHexValue
          RETURN
       END IF
    END IF
 END IF
 RETURN

' reset hilight bytes for mouse.
ResetMouseBytes:
 IF CopyPositionStart = False THEN
    CALL locatecursor2
    RETURN
 END IF
 CALL ResetBytes
 CALL ClearPageByte
 CALL displaypagebyte
 RETURN

' critical error trap.
Error.Routine:
 ' check error return type.
 ErrorTrap = ERR
 IF Check.Disk THEN
    Disk.Ready = ERR
    RESUME NEXT
 END IF
 ' check screen.
 IF ScreenDrawn THEN
    CLS
 END IF
 ScreenDrawn = False
 ' display error message.
 COLORf White
 LOCATEf CSRLIN, 1, 0
 PRINTf "Hex Editor Main " + Version + " " + Release + " critical error trap:"
 ' display error.
 COLORf Yellow
 LOCATEf CSRLIN + 1, 1, 0
 CALL DisplayCriticalError(ErrorTrap)
 ' display error prompt.
 COLORf Green
 LOCATEf CSRLIN + 1, 1, 0
 PRINTf "Press R(etry), C(ontinue), Q(uit), A(bort):"
 LOCATEf CSRLIN, 44, 1
 ' get keypress.
 DO
    CALL Error.Inkey(ErrorRespond$)
    ' parse key.
    SELECT CASE LCASE$(ErrorRespond$)
    CASE "r"
       PRINTf "r"
       LOCATEf CSRLIN + 1, 1, 0
       RESUME
    CASE "c"
       PRINTf "c"
       LOCATEf CSRLIN + 1, 1, 0
       GOSUB RedrawScreen1
       RESUME NEXT
    CASE "q"
       PRINTf "q"
       LOCATEf CSRLIN + 1, 1, 0
       IF CurrentFile = False THEN
          GOSUB RedrawScreen1
          RESUME NEXT
       END IF
       RESUME TopProgram
    CASE "a"
       PRINTf "a"
       RESUME Stoploop
    END SELECT
    R = ReleaseTime
 LOOP
 ' this wont happen..
 CALL ResetTrap
 CALL Delete.Process
 END 2

' end of program.
Stoploop:
 ' close all files.
 GOSUB CloseAllFiles
 GOSUB ResetAllAttributes

' stop program.
StopProgram:
 ON ERROR RESUME NEXT
 COLORf2 7, 0

 ' restore current drive.
 InregsX.AX = &HE00
 InregsX.DX = Drive.Number ' 0=a, 1=b, ..
 CALL InterruptX(&H21, InregsX, OutregsX)

 ' restore current directory.
 ASCIIZ3 = Directory.ASCIIZ
 IF Windows.Detected THEN
    InregsX.AX = &H713B
    InregsX.DS = VARSEG(ASCIIZ3)
    InregsX.DX = VARPTR(ASCIIZ3)
    CALL InterruptX(&H21, InregsX, OutregsX)
 ELSE
    InregsX.AX = &H3B00
    InregsX.DS = VARSEG(ASCIIZ3)
    InregsX.DX = VARPTR(ASCIIZ3)
    CALL InterruptX(&H21, InregsX, OutregsX)
 END IF

 ' restore break service.
 CALL ResetTrap

 ' delete process filenames.
 CALL Delete.Process

 ' terminate program.
 Colorf2 Plain, 0
 Locate ,,1
 END 0

' terminal error exits here.
Error.Exit:
 ' display error.
 COLORf2 White, 0
 CLS
 PRINTf "Hex Editor " + Version + " " + Release + " critical error trap:"
 ' display error and exit.
 COLORf Yellow
 LOCATEf CSRLIN + 1, 1, 0
 CALL DisplayCriticalError(ErrorTrap)
 COLORf Green
 LOCATEf CSRLIN + 1, 1, 0
 PRINTf "Close Hexedit processes, Check TEMP variables, .CFG file,"
 LOCATEf CSRLIN + 1, 1, 0
 PRINTf "or DOS file handles, then restart."
 IF ErrorTrap = 90 THEN ' too many processe.
    PRINTf " Also run delhex.bat"
 END IF
 COLORf Plain
 CALL ResetTrap
 ' delete process filenames.
 CALL Delete.Process
 END 2

' Assign PSP file handles.
Sub Increase.Files
 On Local Error Goto Error.Trap2
 Close
 ' get psp segment
 InregsX.AX=&H6200
 Call InterruptX(&H21,InregsX,OutregsX)
 PSP.Segment=OutregsX.BX
 Def Seg=PSP.Segment
 Command.Line=&H80
 Command.Tail=Command.Line+30
 For Var=1 To 94 ' 90 file handles
    Poke Command.Tail+Var-1,&HFF
 Next
 For Var=1 To 4 ' DOS handles
    FileHandle=Peek(&H18+Var-1)
    Poke Command.Tail+Var-1,FileHandle
 Next
 Poke &H32,94
 Poke &H34,Command.Tail
 Def Seg
 Close
Error.Resume2:
 Exit Sub
Error.Trap2:
 Resume Error.Resume2
End Sub

REM End-of-subprogram. I want 2 black siamese cats.

